#!/bin/bash

# Script to standardize permission formats in the codebase
# This script will find and replace space-separated permissions with hyphenated ones
# It will generate a report of all changes made

# Set the base directory to search in
BASE_DIR="/Users/abdessamad/Documents/pos-system/app"

# Create a backup directory
BACKUP_DIR="/Users/abdessamad/Documents/pos-system/permission_backup_$(date +%Y%m%d_%H%M%S)"
mkdir -p "$BACKUP_DIR"
echo "Created backup directory: $BACKUP_DIR"

# Create a report file
REPORT_FILE="/Users/abdessamad/Documents/pos-system/permission_changes_report.txt"
echo "Permission Format Standardization Report - $(date)" > "$REPORT_FILE"
echo "=================================================" >> "$REPORT_FILE"
echo "" >> "$REPORT_FILE"

# Define permission mappings (space-separated to hyphenated)
# Format: OLD_PERM:NEW_PERM
PERMISSION_MAPPINGS=(
    "'process sales returns':'process-sales-returns'"
    "\"process sales returns\":\"process-sales-returns\""
    "'process purchase returns':'process-purchase-returns'"
    "\"process purchase returns\":\"process-purchase-returns\""
    "'manage sales':'manage-sales'"
    "\"manage sales\":\"manage-sales\""
    "'manage purchases':'manage-purchases'"
    "\"manage purchases\":\"manage-purchases\""
    "'manage clients':'manage-clients'"
    "\"manage clients\":\"manage-clients\""
    "'manage suppliers':'manage-suppliers'"
    "\"manage suppliers\":\"manage-suppliers\""
    "'manage products':'manage-products'"
    "\"manage products\":\"manage-products\""
    "'manage expenses':'manage-expenses'"
    "\"manage expenses\":\"manage-expenses\""
    "'manage users':'manage-users'"
    "\"manage users\":\"manage-users\""
    "'record payments':'record-payments'"
    "\"record payments\":\"record-payments\""
    "'process sales':'process-sales'"
    "\"process sales\":\"process-sales\""
    "'view reports':'view-reports'"
    "\"view reports\":\"view-reports\""
    "'access superadmin panel':'access-superadmin-panel'"
    "\"access superadmin panel\":\"access-superadmin-panel\""
    "'view products':'view-products'"
    "\"view products\":\"view-products\""
    "'create products':'create-products'"
    "\"create products\":\"create-products\""
    "'edit products':'edit-products'"
    "\"edit products\":\"edit-products\""
    "'delete products':'delete-products'"
    "\"delete products\":\"delete-products\""
    "'view sales':'view-sales'"
    "\"view sales\":\"view-sales\""
    "'create sales':'create-sales'"
    "\"create sales\":\"create-sales\""
    "'edit sales':'edit-sales'"
    "\"edit sales\":\"edit-sales\""
    "'delete sales':'delete-sales'"
    "\"delete sales\":\"delete-sales\""
    "'view clients':'view-clients'"
    "\"view clients\":\"view-clients\""
    "'create clients':'create-clients'"
    "\"create clients\":\"create-clients\""
    "'edit clients':'edit-clients'"
    "\"edit clients\":\"edit-clients\""
    "'delete clients':'delete-clients'"
    "\"delete clients\":\"delete-clients\""
    "'view suppliers':'view-suppliers'"
    "\"view suppliers\":\"view-suppliers\""
    "'create suppliers':'create-suppliers'"
    "\"create suppliers\":\"create-suppliers\""
    "'edit suppliers':'edit-suppliers'"
    "\"edit suppliers\":\"edit-suppliers\""
    "'delete suppliers':'delete-suppliers'"
    "\"delete suppliers\":\"delete-suppliers\""
    "'view payments':'view-payments'"
    "\"view payments\":\"view-payments\""
    "'view purchases':'view-purchases'"
    "\"view purchases\":\"view-purchases\""
    "'create purchases':'create-purchases'"
    "\"create purchases\":\"create-purchases\""
    "'edit purchases':'edit-purchases'"
    "\"edit purchases\":\"edit-purchases\""
    "'delete purchases':'delete-purchases'"
    "\"delete purchases\":\"delete-purchases\""
    "'view expenses':'view-expenses'"
    "\"view expenses\":\"view-expenses\""
    "'create expenses':'create-expenses'"
    "\"create expenses\":\"create-expenses\""
    "'edit expenses':'edit-expenses'"
    "\"edit expenses\":\"edit-expenses\""
    "'delete expenses':'delete-expenses'"
    "\"delete expenses\":\"delete-expenses\""
    "'view sales returns':'view-sales-returns'"
    "\"view sales returns\":\"view-sales-returns\""
    "'view purchase returns':'view-purchase-returns'"
    "\"view purchase returns\":\"view-purchase-returns\""
    "'view users':'view-users'"
    "\"view users\":\"view-users\""
    "'create users':'create-users'"
    "\"create users\":\"create-users\""
    "'edit users':'edit-users'"
    "\"edit users\":\"edit-users\""
    "'delete users':'delete-users'"
    "\"delete users\":\"delete-users\""
    "'view dashboard':'view-dashboard'"
    "\"view dashboard\":\"view-dashboard\""
    "'view tenant settings':'view-tenant-settings'"
    "\"view tenant settings\":\"view-tenant-settings\""
    "'edit tenant settings':'edit-tenant-settings'"
    "\"edit tenant settings\":\"edit-tenant-settings\""
    "'manage tenants':'manage-tenants'"
    "\"manage tenants\":\"manage-tenants\""
)

# Function to process a file
process_file() {
    local file="$1"
    local backup_file="$BACKUP_DIR/$(basename "$file")"
    local changes=0
    local content
    
    # Create a backup of the file
    cp "$file" "$backup_file"
    
    # Read the file content
    content=$(cat "$file")
    original_content="$content"
    
    # Apply each permission replacement
    for mapping in "${PERMISSION_MAPPINGS[@]}"; do
        # Split the mapping into old and new permissions
        old_perm="${mapping%%:*}"
        new_perm="${mapping#*:}"
        
        # Check if the file contains this permission
        if grep -q "$old_perm" "$file"; then
            # Count occurrences before replacement
            count_before=$(grep -o "$old_perm" "$file" | wc -l)
            
            # Replace the permission using sed (more compatible)
            sed_old_perm=$(echo "$old_perm" | sed 's/\//\\\//g')
            sed_new_perm=$(echo "$new_perm" | sed 's/\//\\\//g')
            content=$(echo "$content" | sed "s/$sed_old_perm/$sed_new_perm/g")
            
            # If replacements were made, log them
            if [ $count_before -gt 0 ]; then
                echo "  - Replaced $old_perm with $new_perm ($count_before occurrences)" >> "$REPORT_FILE"
                changes=$((changes + count_before))
            fi
        fi
    done
    
    # If changes were made, write the content back to the file
    if [ "$original_content" != "$content" ]; then
        echo "$content" > "$file"
        echo "Updated $file ($changes changes)" >> "$REPORT_FILE"
        echo "" >> "$REPORT_FILE"
        return 1  # Changes were made
    else
        return 0  # No changes
    fi
}

# Find all PHP files and process them
echo "Searching for PHP files in $BASE_DIR..."
echo "Files processed:" >> "$REPORT_FILE"
echo "" >> "$REPORT_FILE"

total_files=0
changed_files=0

while IFS= read -r file; do
    echo "Processing $file..."
    total_files=$((total_files + 1))
    
    process_file "$file"
    if [ $? -eq 1 ]; then
        changed_files=$((changed_files + 1))
    fi
done < <(find "$BASE_DIR" -type f -name "*.php")

# Generate summary
echo "" >> "$REPORT_FILE"
echo "Summary:" >> "$REPORT_FILE"
echo "--------" >> "$REPORT_FILE"
echo "Total files processed: $total_files" >> "$REPORT_FILE"
echo "Files changed: $changed_files" >> "$REPORT_FILE"
echo "Backup directory: $BACKUP_DIR" >> "$REPORT_FILE"

echo ""
echo "Permission standardization complete!"
echo "Total files processed: $total_files"
echo "Files changed: $changed_files"
echo "See detailed report in: $REPORT_FILE"
echo "Backup files are in: $BACKUP_DIR"
