<?php

use Illuminate\Foundation\Application;
use Illuminate\Support\Facades\Route;
use Inertia\Inertia;
use App\Http\Controllers\Web\ProductController;
use App\Http\Controllers\Web\ClientController;
use App\Http\Controllers\Web\SupplierController;
use App\Http\Controllers\Tenant\POSController;

/*
|--------------------------------------------------------------------------
| Web Routes
|--------------------------------------------------------------------------
|
| Here is where you can register web routes for your application. These
| routes are loaded by the RouteServiceProvider within a group which
| contains the "web" middleware group. Now create something great!
|
*/

Route::get('/', function () {
    return Inertia::render('Welcome', [
        'canLogin' => Route::has('login'),
        'canRegister' => Route::has('register'),
        'laravelVersion' => Application::VERSION,
        'phpVersion' => PHP_VERSION,
    ]);
});

Route::middleware([
    'auth:sanctum',
    config('jetstream.auth_session'),
    'verified',
])->group(function () {
    Route::get('/dashboard', [\App\Http\Controllers\DashboardController::class, 'index'])->name('dashboard');
    
    // Product routes with granular permissions
    // View product routes
    Route::middleware(['permission:view products|manage products|view-products|manage-products'])->group(function () {
        Route::get('products', [ProductController::class, 'index'])->name('products.index');
    });
    
    // Create product routes
    Route::middleware(['permission:create products|manage products|create-products|manage-products'])->group(function () {
        Route::get('products/create', [ProductController::class, 'create'])->name('products.create');
        Route::post('products', [ProductController::class, 'store'])->name('products.store');
    });
    
    // View single product route
    Route::middleware(['permission:view products|manage products|view-products|manage-products'])->group(function () {
        Route::get('products/{product}', [ProductController::class, 'show'])->name('products.show');
    });
    
    // Edit product routes
    Route::middleware(['permission:edit products|manage products|edit-products|manage-products'])->group(function () {
        Route::get('products/{product}/edit', [ProductController::class, 'edit'])->name('products.edit');
        Route::put('products/{product}', [ProductController::class, 'update'])->name('products.update');
        Route::patch('products/{product}', [ProductController::class, 'update']);
    });
    
    // Delete product routes
    Route::middleware(['permission:delete products|manage products|delete-products|manage-products'])->group(function () {
        Route::delete('products/{product}', [ProductController::class, 'destroy'])->name('products.destroy');
    });
    
    // Category routes with product management permission
    Route::middleware(['permission:manage products|manage-products'])->group(function () {
        Route::resource('categories', App\Http\Controllers\Tenant\CategoryController::class);
    });
    
    // Inventory routes - require specific product management permission
    Route::middleware(['permission:manage products|manage-products'])->group(function () {
        Route::resource('stock-adjustments', App\Http\Controllers\Tenant\StockAdjustmentController::class)
            ->only(['index', 'create', 'store']);
    });
    
    // Product search now uses API route
    
    // Client routes - protected by middleware for proper access control
    // IMPORTANT: Define specific routes BEFORE wildcard routes to prevent conflicts
    
    // Create client routes - must be BEFORE {client} wildcard routes
    Route::middleware(['permission:create clients|manage clients|create-clients|manage-clients'])->group(function () {
        Route::get('clients/create', [ClientController::class, 'create'])->name('clients.create');
        Route::post('clients', [ClientController::class, 'store'])->name('clients.store');
    });
    
    // Edit client routes - specific routes with {client} parameter
    Route::middleware(['permission:edit clients|manage clients|edit-clients|manage-clients'])->group(function () {
        Route::get('clients/{client}/edit', [ClientController::class, 'edit'])->name('clients.edit');
        Route::put('clients/{client}', [ClientController::class, 'update'])->name('clients.update');
        Route::patch('clients/{client}', [ClientController::class, 'update']);
    });
    
    // View client routes - generic wildcard routes AFTER specific routes
    Route::middleware(['permission:view clients|manage clients|view-clients|manage-clients'])->group(function () {
        Route::get('clients', [ClientController::class, 'index'])->name('clients.index');
        Route::get('clients/{client}', [ClientController::class, 'show'])->name('clients.show');
    });
    
    // Delete client routes
    Route::middleware(['permission:delete clients|manage clients|delete-clients|manage-clients'])->group(function () {
        Route::delete('clients/{client}', [ClientController::class, 'destroy'])->name('clients.destroy');
    });
    
    // Supplier routes - protected by middleware for proper access control
    // IMPORTANT: Define specific routes BEFORE wildcard routes to prevent conflicts
    Route::middleware(['permission:view suppliers|manage suppliers|view-suppliers|manage-suppliers'])->group(function () {
        Route::get('suppliers', [SupplierController::class, 'index'])->name('suppliers.index');
    });
    
    // Create supplier routes - must be BEFORE {supplier} wildcard routes
    Route::middleware(['permission:create suppliers|manage suppliers|create-suppliers|manage-suppliers'])->group(function () {
        Route::get('suppliers/create', [SupplierController::class, 'create'])->name('suppliers.create');
        Route::post('suppliers', [SupplierController::class, 'store'])->name('suppliers.store');
    });
    
    // View supplier detail route - needs to come after specific routes
    Route::middleware(['permission:view suppliers|manage suppliers|view-suppliers|manage-suppliers'])->group(function () {
        Route::get('suppliers/{supplier}', [SupplierController::class, 'show'])->name('suppliers.show');
    });
    
    Route::middleware(['permission:edit suppliers|manage suppliers|edit-suppliers|manage-suppliers'])->group(function () {
        Route::get('suppliers/{supplier}/edit', [SupplierController::class, 'edit'])->name('suppliers.edit');
        Route::put('suppliers/{supplier}', [SupplierController::class, 'update'])->name('suppliers.update');
        Route::patch('suppliers/{supplier}', [SupplierController::class, 'update']);
    });
    
    Route::middleware(['permission:delete suppliers|manage suppliers|delete-suppliers|manage-suppliers'])->group(function () {
        Route::delete('suppliers/{supplier}', [SupplierController::class, 'destroy'])->name('suppliers.destroy');
    });
    

    
    // Payment routes - protected by middleware for proper access control
    Route::middleware(['permission:record payments|manage payments|record-payments|manage-payments'])->group(function () {
        Route::get('payments/create', [\App\Http\Controllers\Web\PaymentController::class, 'create'])->name('payments.create');
        Route::post('payments/sale', [\App\Http\Controllers\Web\PaymentController::class, 'storeSalePayment'])->name('payments.sale.store');
        Route::post('clients/{client}/payments', [\App\Http\Controllers\Web\PaymentController::class, 'storeClientPayment'])->name('clients.payments.store');
        Route::post('suppliers/{supplier}/payments', [\App\Http\Controllers\Web\PaymentController::class, 'storeSupplierPayment'])->name('suppliers.payments.store');
    });

    // Checks Management
    Route::middleware(['permission:view payments|manage payments|view-payments|manage-payments'])->group(function () {
        Route::get('checks-management', [\App\Http\Controllers\Web\ChecksManagementController::class, 'index'])->name('checks-management.index');
        
        // Sales Checks
        Route::get('checks', [\App\Http\Controllers\Web\CheckController::class, 'index'])->name('checks.index');
        Route::patch('checks/{payment}/status', [\App\Http\Controllers\Web\CheckController::class, 'updateStatus'])->name('checks.update-status');
        Route::get('api/checks/due', [\App\Http\Controllers\Web\CheckController::class, 'getDueChecks'])->name('api.checks.due');
        
        // Supplier Checks
        Route::get('supplier-checks', [\App\Http\Controllers\Web\SupplierCheckController::class, 'index'])->name('supplier-checks.index');
        Route::patch('supplier-checks/{payment}/status', [\App\Http\Controllers\Web\SupplierCheckController::class, 'updateStatus'])->name('supplier-checks.update-status');
        Route::get('api/supplier-checks/due', [\App\Http\Controllers\Web\SupplierCheckController::class, 'getDueChecks'])->name('api.supplier-checks.due');
    });

    // Debug route to test User model
    Route::get('/debug/user', function() {
        $user = \App\Models\User::first();
        try {
            return response()->json([
                'user' => $user->toArray(),
                'profile_photo_path' => $user->profile_photo_path,
                'profile_photo_url' => $user->profile_photo_url
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'error' => $e->getMessage(),
                'file' => $e->getFile(),
                'line' => $e->getLine()
            ], 500);
        }
    });

    // Sales routes - sales list and details are accessible to all authenticated users
    // but create, edit, and delete operations require sales permissions
    Route::get('/sales', [\App\Http\Controllers\Web\SaleController::class, 'index'])->name('sales.index');
    
    // Updated to accept both create sales and process sales permissions
    Route::middleware(['permission:create sales|process sales|create-sales|process-sales'])->group(function () {
        Route::get('/sales/create', [\App\Http\Controllers\Web\SaleController::class, 'create'])->name('sales.create');
        Route::post('/sales', [\App\Http\Controllers\Web\SaleController::class, 'store'])->name('sales.store');
    });
    
    // Sale Returns routes - accessible to all authenticated users
    Route::get('/sales/returns', [\App\Http\Controllers\Web\SaleReturnController::class, 'index'])->name('sales.returns.index');
    Route::get('/sales/returns/{saleReturn}', [\App\Http\Controllers\Web\SaleReturnController::class, 'show'])->name('sales.returns.show');
    
    // New route for selecting a sale to return
    Route::get('/sales/returns/create/select', [\App\Http\Controllers\Web\SaleReturnController::class, 'selectSale'])->name('sales.returns.select');
    
    // Purchase Returns routes - accessible to users with process purchase returns permission
    Route::middleware(['permission:process purchase returns|process-purchase-returns'])->group(function () {
        Route::get('/purchases/returns', [\App\Http\Controllers\Web\PurchaseReturnController::class, 'index'])->name('purchases.returns.index');
        Route::get('/purchases/returns/{purchaseReturn}', [\App\Http\Controllers\Web\PurchaseReturnController::class, 'show'])->name('purchases.returns.show');
        
        // New route for selecting a purchase to return
        Route::get('/purchases/returns/create/select', [\App\Http\Controllers\Web\PurchaseReturnController::class, 'selectPurchase'])->name('purchases.returns.select');
    });
    
    // Receipt route for sales - requires process sales permission
    Route::get('/sales/{sale}/receipt/html', [\App\Http\Controllers\Tenant\ReceiptController::class, 'html'])
        ->name('sales.receipt.html')
        ->middleware(['auth', 'can:process sales']);
        
    // These routes must come after the /sales/create route to avoid conflicts
    Route::get('/sales/{sale}', [\App\Http\Controllers\Web\SaleController::class, 'show'])->name('sales.show');
    
    Route::middleware(['permission:edit sales|edit-sales|manage sales|manage-sales'])->group(function () {
        Route::get('/sales/{sale}/edit', [\App\Http\Controllers\Web\SaleController::class, 'edit'])->name('sales.edit');
        Route::put('/sales/{sale}', [\App\Http\Controllers\Web\SaleController::class, 'update'])->name('sales.update');
    });
    
    Route::middleware(['permission:delete sales|delete-sales|manage sales|manage-sales'])->group(function () {
        Route::delete('/sales/{sale}', [\App\Http\Controllers\Web\SaleController::class, 'destroy'])->name('sales.destroy');
    });
    
    // PDF Invoice Download route
    Route::middleware(['permission:process-sales|process sales|view-sales|view sales'])->group(function () {
        Route::get('/sales/{sale}/invoice/pdf', [\App\Http\Controllers\Tenant\InvoiceController::class, 'download'])->name('sales.invoice.download');
    });
    
    // Sale finalization now uses API route
    
    // Purchase routes with granular permissions
    // Purchase routes with granular permissions
    // View purchase routes
    Route::middleware(['permission:view purchases|manage purchases|view-purchases|manage-purchases'])->group(function () {
        Route::get('purchases', [\App\Http\Controllers\Web\PurchaseController::class, 'index'])->name('purchases.index');
    });
    
    // Create purchase routes
    Route::middleware(['permission:create purchases|manage purchases|create-purchases|manage-purchases'])->group(function () {
        Route::get('purchases/create', [\App\Http\Controllers\Web\PurchaseController::class, 'create'])->name('purchases.create');
        Route::post('purchases', [\App\Http\Controllers\Web\PurchaseController::class, 'store'])->name('purchases.store');
    });
    
    // View single purchase route - moved after the create route to fix route order
    Route::middleware(['permission:view purchases|manage purchases|view-purchases|manage-purchases'])->group(function () {
        Route::get('purchases/{purchase}', [\App\Http\Controllers\Web\PurchaseController::class, 'show'])->name('purchases.show');
    });
    
    // Edit purchase routes
    Route::middleware(['permission:edit purchases|manage purchases|edit-purchases|manage-purchases'])->group(function () {
        Route::get('purchases/{purchase}/edit', [\App\Http\Controllers\Web\PurchaseController::class, 'edit'])->name('purchases.edit');
        Route::put('purchases/{purchase}', [\App\Http\Controllers\Web\PurchaseController::class, 'update'])->name('purchases.update');
        Route::patch('purchases/{purchase}', [\App\Http\Controllers\Web\PurchaseController::class, 'update']);
    });
    
    // Delete purchase routes
    Route::middleware(['permission:delete purchases|manage purchases|delete-purchases|manage-purchases'])->group(function () {
        Route::delete('purchases/{purchase}', [\App\Http\Controllers\Web\PurchaseController::class, 'destroy'])->name('purchases.destroy');
    });
    
    // Expense routes with strict permission enforcement
    // View expenses - requires explicit 'view expenses' or 'manage expenses' permission
    Route::middleware(['strict.expense:view expenses,manage expenses'])->group(function () {
        Route::get('expenses', [\App\Http\Controllers\Web\ExpenseController::class, 'index'])->name('expenses.index');
        Route::get('expenses/{expense}', [\App\Http\Controllers\Web\ExpenseController::class, 'show'])->name('expenses.show');
    });
    
    // Create expenses - requires explicit 'create expenses' or 'manage expenses' permission
    Route::middleware(['strict.expense:create expenses,manage expenses'])->group(function () {
        Route::get('expenses/create', [\App\Http\Controllers\Web\ExpenseController::class, 'create'])->name('expenses.create');
        Route::post('expenses', [\App\Http\Controllers\Web\ExpenseController::class, 'store'])->name('expenses.store');
    });
    
    // Edit expenses - requires explicit 'edit expenses' or 'manage expenses' permission
    Route::middleware(['strict.expense:edit expenses,manage expenses'])->group(function () {
        Route::get('expenses/{expense}/edit', [\App\Http\Controllers\Web\ExpenseController::class, 'edit'])->name('expenses.edit');
        Route::put('expenses/{expense}', [\App\Http\Controllers\Web\ExpenseController::class, 'update'])->name('expenses.update');
        Route::patch('expenses/{expense}', [\App\Http\Controllers\Web\ExpenseController::class, 'update']);
    });
    
    // Delete expenses - requires explicit 'delete expenses' or 'manage expenses' permission
    Route::middleware(['strict.expense:delete expenses,manage expenses'])->group(function () {
        Route::delete('expenses/{expense}', [\App\Http\Controllers\Web\ExpenseController::class, 'destroy'])->name('expenses.destroy');
    });
    
    // Tenant User Management routes - accessible only to tenant admins
    Route::middleware(['auth', 'tenant.admin'])->prefix('tenant')->name('tenant.')->group(function () {
        Route::resource('users', \App\Http\Controllers\Tenant\UserController::class);
        // Tenant settings routes
        Route::get('settings', [\App\Http\Controllers\Tenant\SettingsController::class, 'edit'])
            ->name('settings.edit')
            ->middleware(['can:manage tenant settings']);
        Route::post('settings', [\App\Http\Controllers\Tenant\SettingsController::class, 'update'])
            ->name('settings.update')
            ->middleware(['can:manage tenant settings']);
    });
    
    // Report routes - accessible only to users with view reports permission
    Route::middleware(['permission:view reports|view-reports'])->group(function () {
        Route::get('/reports', [\App\Http\Controllers\Web\ReportController::class, 'index'])->name('reports.index');
        Route::get('/reports/sales', [\App\Http\Controllers\Web\ReportController::class, 'salesReport'])->name('reports.sales');
        Route::get('/reports/inventory', [\App\Http\Controllers\Web\ReportController::class, 'inventoryReport'])->name('reports.inventory');
        Route::get('/reports/expenses', [\App\Http\Controllers\Web\ReportController::class, 'expenseReport'])->name('reports.expenses');
        Route::get('/reports/purchases', [\App\Http\Controllers\Web\ReportController::class, 'purchaseReport'])->name('reports.purchases');
        Route::get('/reports/sales-returns', [\App\Http\Controllers\Web\ReportController::class, 'salesReturnsReport'])->name('reports.sales-returns');
        Route::get('/reports/purchase-returns', [\App\Http\Controllers\Web\ReportController::class, 'purchaseReturnsReport'])->name('reports.purchase-returns');
        
        // Profit Reports
        Route::get('/reports/profit/sales', [\App\Http\Controllers\Tenant\ProfitReportController::class, 'salesProfit'])->name('reports.profit.sales');
        Route::get('/reports/profit/products', [\App\Http\Controllers\Tenant\ProfitReportController::class, 'productsProfit'])->name('reports.profit.products');
    });
});

// Superadmin routes - only accessible to users with 'access superadmin panel' permission

// POS Routes
Route::middleware([
    'auth:sanctum',
    config('jetstream.auth_session'),
    'verified',
    'permission:process-sales'
])->group(function () {
    Route::get('/pos', [POSController::class, 'index'])->name('pos.index');
    Route::get('/api/pos/products/search', [POSController::class, 'searchProducts'])->name('api.pos.products.search');
    Route::post('/api/pos/finalize-sale', [POSController::class, 'finalizeSale'])->name('api.pos.finalize-sale');
    Route::get('/sales/receipt/{id}', [POSController::class, 'receipt'])->name('sales.receipt');
});

// TEMP: Test route for troubleshooting 404 on /clients/create
Route::get('/clients/create-test', function () {
    return 'Test route works!';
});
Route::middleware([
    'auth:sanctum',
    config('jetstream.auth_session'),
    'verified',
    'permission:access superadmin panel|access-superadmin-panel'
])->prefix('superadmin')->name('superadmin.')->group(function () {
    Route::resource('tenants', \App\Http\Controllers\Superadmin\TenantController::class);
    
    // Platform-wide Activity Log
    Route::get('/activity-logs', [\App\Http\Controllers\Superadmin\ActivityLogController::class, 'index'])
        ->name('activity-log.index');
});
