import axios from 'axios';

// First, get the CSRF cookie - this is crucial for Sanctum authentication
const initCsrf = async () => {
    try {
        // This endpoint sets the CSRF cookie that Sanctum needs
        await axios.get('/sanctum/csrf-cookie', { withCredentials: true });
        console.log('CSRF cookie initialized');
        return true;
    } catch (error) {
        console.error('Failed to initialize CSRF cookie:', error);
        return false;
    }
};

// Initialize CSRF token immediately when this file is imported
// This ensures the CSRF token is available before any API requests are made
const csrfPromise = initCsrf();

// Create a custom axios instance for API requests
const api = axios.create({
    baseURL: '/api',
    withCredentials: true,
    headers: {
        'X-Requested-With': 'XMLHttpRequest',
        'Content-Type': 'application/json',
        'Accept': 'application/json',
    }
});

// Add a request interceptor to ensure CSRF token is set
api.interceptors.request.use(async (config) => {
    // Wait for the CSRF token to be initialized
    await csrfPromise;
    
    // Get the CSRF token from the meta tag
    const token = document.head.querySelector('meta[name="csrf-token"]');
    if (token) {
        config.headers['X-CSRF-TOKEN'] = token.content;
    }
    
    // Get the XSRF-TOKEN from cookies
    const xsrfToken = getCookie('XSRF-TOKEN');
    if (xsrfToken) {
        config.headers['X-XSRF-TOKEN'] = decodeURIComponent(xsrfToken);
    }
    
    return config;
});

// Helper function to get cookie value by name
function getCookie(name) {
    const value = `; ${document.cookie}`;
    const parts = value.split(`; ${name}=`);
    if (parts.length === 2) return parts.pop().split(';').shift();
}

// Add a response interceptor to handle authentication errors
api.interceptors.response.use(
    response => response,
    async error => {
        if (error.response && error.response.status === 401) {
            console.error('Authentication error:', error);
            // Optionally redirect to login page or refresh token
            // window.location.href = '/login';
        }
        return Promise.reject(error);
    }
);

export default api;
