<?php

namespace App\Http\Controllers;

use App\Models\Supplier;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class SupplierController extends Controller
{
    /**
     * Create a new controller instance.
     */
    public function __construct()
    {
        $this->middleware('permission:manage suppliers')->except(['index', 'show']);
    }
    
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $suppliers = Supplier::where('tenant_id', Auth::user()->tenant_id)
            ->orderBy('name')
            ->paginate(15);
            
        return response()->json($suppliers);
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'nullable|email|max:255|unique:suppliers,email,NULL,id,tenant_id,' . Auth::user()->tenant_id,
            'phone' => 'nullable|string|max:20',
            'address' => 'nullable|string|max:255',
            'notes' => 'nullable|string',
        ]);
        
        $supplier = new Supplier([
            'tenant_id' => Auth::user()->tenant_id,
            'name' => $request->name,
            'email' => $request->email,
            'phone' => $request->phone,
            'address' => $request->address,
            'notes' => $request->notes,
            'balance' => 0, // New suppliers start with zero balance
        ]);
        
        $supplier->save();
        
        return response()->json($supplier, 201);
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        $supplier = Supplier::where('id', $id)
            ->where('tenant_id', Auth::user()->tenant_id)
            ->first();
            
        if (!$supplier) {
            return response()->json(['message' => 'Supplier not found'], 404);
        }
        
        return response()->json($supplier);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id)
    {
        $supplier = Supplier::where('id', $id)
            ->where('tenant_id', Auth::user()->tenant_id)
            ->first();
            
        if (!$supplier) {
            return response()->json(['message' => 'Supplier not found'], 404);
        }
        
        $request->validate([
            'name' => 'sometimes|required|string|max:255',
            'email' => 'nullable|email|max:255|unique:suppliers,email,' . $supplier->id . ',id,tenant_id,' . Auth::user()->tenant_id,
            'phone' => 'nullable|string|max:20',
            'address' => 'nullable|string|max:255',
            'notes' => 'nullable|string',
        ]);
        
        // Update supplier fields
        if ($request->filled('name')) $supplier->name = $request->name;
        if ($request->filled('email')) $supplier->email = $request->email;
        if ($request->filled('phone')) $supplier->phone = $request->phone;
        if ($request->filled('address')) $supplier->address = $request->address;
        if ($request->filled('notes')) $supplier->notes = $request->notes;
        
        $supplier->save();
        
        return response()->json($supplier);
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        $supplier = Supplier::where('id', $id)
            ->where('tenant_id', Auth::user()->tenant_id)
            ->first();
            
        if (!$supplier) {
            return response()->json(['message' => 'Supplier not found'], 404);
        }
        
        // Check if supplier has any payments
        $paymentsCount = $supplier->payments()->count();
        if ($paymentsCount > 0) {
            return response()->json([
                'message' => 'Cannot delete supplier with associated payments',
                'payments_count' => $paymentsCount
            ], 422);
        }
        
        // Check if supplier has outstanding balance
        if ($supplier->balance > 0) {
            return response()->json([
                'message' => 'Cannot delete supplier with outstanding balance',
                'balance' => $supplier->balance
            ], 422);
        }
        
        $supplier->delete();
        
        return response()->json(null, 204);
    }
}
