<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use Spatie\Permission\Models\Permission;
use Spatie\Permission\Models\Role;
use App\Models\User;
use Illuminate\Support\Facades\DB;

class StandardizePermissionFormat extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'permissions:standardize-format {--dry-run : Run in dry-run mode without making changes}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Standardize all permissions to use hyphenated format and remove duplicates';

    /**
     * Permission mapping from space-separated to hyphenated format
     */
    protected $permissionMapping = [
        'process sales returns' => 'process-sales-returns',
        'process purchase returns' => 'process-purchase-returns',
        'manage sales' => 'manage-sales',
        'manage purchases' => 'manage-purchases',
        'manage clients' => 'manage-clients',
        'manage suppliers' => 'manage-suppliers',
        'manage products' => 'manage-products',
        'manage expenses' => 'manage-expenses',
        'manage users' => 'manage-users',
        'record payments' => 'record-payments',
        'process sales' => 'process-sales',
        'view reports' => 'view-reports',
        'access superadmin panel' => 'access-superadmin-panel',
    ];

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $dryRun = $this->option('dry-run');
        
        if ($dryRun) {
            $this->info('Running in dry-run mode. No changes will be made.');
        }
        
        $this->info('Starting permission format standardization...');
        
        // Step 1: Ensure all hyphenated permissions exist
        $this->ensureHyphenatedPermissionsExist($dryRun);
        
        // Step 2: Update role_has_permissions to use hyphenated permissions
        $this->updateRolePermissions($dryRun);
        
        // Step 3: Update model_has_permissions to use hyphenated permissions
        $this->updateUserPermissions($dryRun);
        
        // Step 4: Remove duplicate space-separated permissions
        if (!$dryRun) {
            $this->removeSpaceSeparatedPermissions();
        } else {
            $this->info('[DRY RUN] Would remove space-separated permissions');
        }
        
        // Step 5: Clear permission cache
        if (!$dryRun) {
            app()['cache']->forget('spatie.permission.cache');
            $this->info('Permission cache cleared.');
        } else {
            $this->info('[DRY RUN] Would clear permission cache');
        }
        
        $this->info('Permission format standardization completed successfully!');
        
        return 0;
    }
    
    /**
     * Ensure all hyphenated permissions exist
     */
    protected function ensureHyphenatedPermissionsExist($dryRun)
    {
        $this->info('Ensuring all hyphenated permissions exist...');
        
        foreach ($this->permissionMapping as $oldFormat => $newFormat) {
            $exists = Permission::where('name', $newFormat)->exists();
            
            if (!$exists) {
                if (!$dryRun) {
                    Permission::create(['name' => $newFormat]);
                    $this->line("  Created permission: {$newFormat}");
                } else {
                    $this->line("  [DRY RUN] Would create permission: {$newFormat}");
                }
            } else {
                $this->line("  Permission already exists: {$newFormat}");
            }
        }
    }
    
    /**
     * Update role permissions to use hyphenated format
     */
    protected function updateRolePermissions($dryRun)
    {
        $this->info('Updating role permissions to use hyphenated format...');
        
        $roles = Role::all();
        $this->line("Found {$roles->count()} roles to update.");
        
        foreach ($roles as $role) {
            $this->line("Processing role: {$role->name}");
            $rolePermissions = $role->permissions->pluck('name')->toArray();
            
            foreach ($this->permissionMapping as $oldFormat => $newFormat) {
                if (in_array($oldFormat, $rolePermissions) && !in_array($newFormat, $rolePermissions)) {
                    if (!$dryRun) {
                        // Get the permission models
                        $oldPermission = Permission::where('name', $oldFormat)->first();
                        $newPermission = Permission::where('name', $newFormat)->first();
                        
                        if ($oldPermission && $newPermission) {
                            // Find the role_has_permissions entry
                            $roleHasPermission = DB::table('role_has_permissions')
                                ->where('permission_id', $oldPermission->id)
                                ->where('role_id', $role->id)
                                ->first();
                            
                            if ($roleHasPermission) {
                                // Add the new permission
                                DB::table('role_has_permissions')->insert([
                                    'permission_id' => $newPermission->id,
                                    'role_id' => $role->id
                                ]);
                                
                                $this->line("  Added {$newFormat} to role {$role->name}");
                            }
                        }
                    } else {
                        $this->line("  [DRY RUN] Would add {$newFormat} to role {$role->name}");
                    }
                }
            }
        }
    }
    
    /**
     * Update user permissions to use hyphenated format
     */
    protected function updateUserPermissions($dryRun)
    {
        $this->info('Updating user permissions to use hyphenated format...');
        
        $users = User::all();
        $this->line("Found {$users->count()} users to check.");
        
        foreach ($users as $user) {
            $this->line("Processing user: {$user->name} (ID: {$user->id})");
            $userPermissions = $user->getDirectPermissions()->pluck('name')->toArray();
            
            foreach ($this->permissionMapping as $oldFormat => $newFormat) {
                if (in_array($oldFormat, $userPermissions) && !in_array($newFormat, $userPermissions)) {
                    if (!$dryRun) {
                        // Get the permission models
                        $oldPermission = Permission::where('name', $oldFormat)->first();
                        $newPermission = Permission::where('name', $newFormat)->first();
                        
                        if ($oldPermission && $newPermission) {
                            // Find the model_has_permissions entry
                            $modelHasPermission = DB::table('model_has_permissions')
                                ->where('permission_id', $oldPermission->id)
                                ->where('model_id', $user->id)
                                ->where('model_type', get_class($user))
                                ->first();
                            
                            if ($modelHasPermission) {
                                // Add the new permission
                                DB::table('model_has_permissions')->insert([
                                    'permission_id' => $newPermission->id,
                                    'model_id' => $user->id,
                                    'model_type' => get_class($user)
                                ]);
                                
                                $this->line("  Added {$newFormat} to user {$user->name}");
                            }
                        }
                    } else {
                        $this->line("  [DRY RUN] Would add {$newFormat} to user {$user->name}");
                    }
                }
            }
        }
    }
    
    /**
     * Remove space-separated permissions
     */
    protected function removeSpaceSeparatedPermissions()
    {
        $this->info('Removing space-separated permissions...');
        
        foreach ($this->permissionMapping as $oldFormat => $newFormat) {
            $oldPermission = Permission::where('name', $oldFormat)->first();
            
            if ($oldPermission) {
                // Remove from role_has_permissions
                $count = DB::table('role_has_permissions')
                    ->where('permission_id', $oldPermission->id)
                    ->delete();
                $this->line("  Removed {$count} role assignments for {$oldFormat}");
                
                // Remove from model_has_permissions
                $count = DB::table('model_has_permissions')
                    ->where('permission_id', $oldPermission->id)
                    ->delete();
                $this->line("  Removed {$count} user assignments for {$oldFormat}");
                
                // Delete the permission
                $oldPermission->delete();
                $this->line("  Deleted permission: {$oldFormat}");
            } else {
                $this->line("  Permission not found: {$oldFormat}");
            }
        }
    }
}
