<?php

namespace App\Http\Controllers;

use App\Http\Requests\StoreProductRequest;
use App\Http\Requests\UpdateProductRequest;
use App\Models\Product;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class ProductController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        // Check if this is a search request
        if ($request->has('search')) {
            $search = $request->input('search');
            
            // Get products matching the search query for the current tenant
            $products = Product::where('tenant_id', Auth::user()->tenant_id)
                ->where(function($query) use ($search) {
                    $query->where('name', 'like', "%{$search}%")
                          ->orWhere('sku', 'like', "%{$search}%");
                })
                ->select('id', 'name', 'sku', 'price as selling_price', 'quantity as stock_quantity')
                ->limit(10)
                ->get();
                
            return response()->json(['data' => $products]);
        }
        
        // Regular index request - get all products for the current tenant with pagination
        $products = Product::where('tenant_id', Auth::user()->tenant_id)
            ->orderBy('name')
            ->paginate(15);
            
        return response()->json($products);
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(StoreProductRequest $request)
    {
        // Create a new product for the current tenant
        $product = new Product($request->validated());
        $product->tenant_id = Auth::user()->tenant_id;
        $product->save();
        
        return response()->json($product, 201);
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        // Find the product belonging to the current tenant
        $product = Product::where('id', $id)
            ->where('tenant_id', Auth::user()->tenant_id)
            ->first();
            
        if (!$product) {
            return response()->json(['message' => 'Product not found'], 404);
        }
        
        return response()->json($product);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(UpdateProductRequest $request, string $id)
    {
        // Find the product belonging to the current tenant
        $product = Product::where('id', $id)
            ->where('tenant_id', Auth::user()->tenant_id)
            ->first();
            
        if (!$product) {
            return response()->json(['message' => 'Product not found'], 404);
        }
        
        // Update the product with validated data
        $product->update($request->validated());
        
        return response()->json($product);
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        // Find the product belonging to the current tenant
        $product = Product::where('id', $id)
            ->where('tenant_id', Auth::user()->tenant_id)
            ->first();
            
        if (!$product) {
            return response()->json(['message' => 'Product not found'], 404);
        }
        
        // Delete the product
        $product->delete();
        
        return response()->json(null, 204);
    }
}
