<?php

namespace App\Console\Commands;

use App\Models\Payment;
use App\Notifications\CheckDueNotification;
use Carbon\Carbon;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Notification;
use Spatie\Multitenancy\Models\Tenant;

class NotifyDueChecks extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'app:notify-due-checks {--days=3 : Number of days before due date to send notification}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Notify users about checks that are due soon';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $daysBeforeDue = $this->option('days');
        $this->info("Checking for checks due in {$daysBeforeDue} days...");
        
        // Process for each tenant
        Tenant::all()->each(function (Tenant $tenant) use ($daysBeforeDue) {
            $tenant->execute(function () use ($tenant, $daysBeforeDue) {
                $this->processChecksForTenant($tenant, $daysBeforeDue);
            });
        });
        
        $this->info('Check notification process completed.');
        
        return Command::SUCCESS;
    }
    
    /**
     * Process checks for a specific tenant.
     */
    protected function processChecksForTenant(Tenant $tenant, int $daysBeforeDue): void
    {
        $this->info("Processing tenant: {$tenant->name}");
        
        // Calculate the target date
        $targetDate = Carbon::now()->addDays($daysBeforeDue)->format('Y-m-d');
        
        // Find all check payments that are due on the target date
        $dueChecks = Payment::where('tenant_id', $tenant->id)
            ->where('method', 'check')
            ->whereNotNull('check_due_date')
            ->whereDate('check_due_date', $targetDate)
            ->with(['payable', 'user'])
            ->get();
            
        if ($dueChecks->isEmpty()) {
            $this->info("No checks due on {$targetDate} for tenant {$tenant->name}");
            return;
        }
        
        $this->info("Found {$dueChecks->count()} checks due on {$targetDate} for tenant {$tenant->name}");
        
        // Process each due check
        foreach ($dueChecks as $check) {
            $this->processCheck($check, $tenant);
        }
    }
    
    /**
     * Process a single due check.
     */
    protected function processCheck(Payment $check, Tenant $tenant): void
    {
        $payableType = class_basename($check->payable_type);
        $payableName = $check->payable->name ?? 'Unknown';
        
        $this->info("Processing check #{$check->id} for {$payableType} '{$payableName}' amount: {$check->amount}");
        
        // Get users to notify (admin and manager roles)
        $usersToNotify = $tenant->users()->whereIn('role', ['admin', 'manager'])->get();
        
        if ($usersToNotify->isEmpty()) {
            $this->warn("No users to notify for tenant {$tenant->name}");
            return;
        }
        
        // Send notifications
        try {
            Notification::send($usersToNotify, new CheckDueNotification($check));
            $this->info("Notification sent to {$usersToNotify->count()} users");
        } catch (\Exception $e) {
            $this->error("Failed to send notification: {$e->getMessage()}");
            Log::error("Failed to send check due notification", [
                'check_id' => $check->id,
                'error' => $e->getMessage(),
            ]);
        }
    }
}
