<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Inertia\Inertia;
use App\Models\Sale;
use App\Models\Client;
use App\Models\Supplier;
use App\Models\Product;
use App\Models\Expense;
use Carbon\Carbon;

class DashboardController extends Controller
{
    /**
     * Create a new controller instance.
     */
    public function __construct()
    {
        $this->middleware('auth');
    }
    
    /**
     * Show the application dashboard.
     *
     * @return \Inertia\Response
     */
    public function index()
    {
        $user = Auth::user();
        $tenantId = $user->tenant_id;
        $today = Carbon::today();
        
        // Get today's sales with their items (excluding soft-deleted sales)
        $todaySales = Sale::with('saleItems')
            ->withoutTrashed() // Exclude soft-deleted sales
            ->where('tenant_id', $tenantId)
            ->whereDate('created_at', $today)
            ->get();
            
        // Calculate total amount for each sale if needed and sum up the totals
        $totalSalesToday = 0;
        foreach ($todaySales as $sale) {
            if ($sale->total_amount == 0 && $sale->saleItems->count() > 0) {
                // Update the sale with the calculated total
                $calculatedTotal = $sale->calculateTotalAmount();
                $sale->total_amount = $calculatedTotal;
                $sale->save();
                $totalSalesToday += $calculatedTotal;
            } else {
                $totalSalesToday += $sale->total_amount;
            }
        }
            
        // Base dashboard data for all users
        $dashboardData = [
            'totalSalesToday' => $totalSalesToday,
            'salesCountToday' => $todaySales->count(),
        ];
        
        // Additional data for admin/manager roles
        if ($user->can('view-reports')) {
            // Outstanding client balances
            $outstandingClientBalances = Client::where('tenant_id', $tenantId)
                ->where('balance', '>', 0)
                ->sum('balance');
            
            // Owed supplier balances
            $owedSupplierBalances = Supplier::where('tenant_id', $tenantId)
                ->where('balance', '>', 0)
                ->sum('balance');
            
            // Low stock items count
            $lowStockItemsCount = Product::where('tenant_id', $tenantId)
                ->whereColumn('quantity', '<=', 'reorder_point')
                ->count();
            
            // Recent expenses (last 7 days)
            $recentExpensesTotal = Expense::where('tenant_id', $tenantId)
                ->where('expense_date', '>=', Carbon::now()->subDays(7))
                ->sum('amount');
            
            // Add admin/manager specific data
            $dashboardData = array_merge($dashboardData, [
                'outstandingClientBalances' => $outstandingClientBalances,
                'owedSupplierBalances' => $owedSupplierBalances,
                'lowStockItemsCount' => $lowStockItemsCount,
                'recentExpensesTotal' => $recentExpensesTotal,
                'recentExpenses' => Expense::where('tenant_id', $tenantId)
                    ->where('expense_date', '>=', Carbon::now()->subDays(7))
                    ->orderBy('expense_date', 'desc')
                    ->limit(5)
                    ->get()
            ]);
        }
        
        // For cashiers, add their personal sales data
        if (!$user->can('view-reports') && $user->can('process-sales')) {
            $dashboardData['mySalesToday'] = Sale::withoutTrashed() // Exclude soft-deleted sales
                ->where('tenant_id', $tenantId)
                ->where('user_id', $user->id) // Assuming user_id is the correct column
                ->whereDate('created_at', $today) // Assuming created_at is the correct column
                ->sum('total_amount');
            
            $dashboardData['mySalesCountToday'] = Sale::withoutTrashed() // Exclude soft-deleted sales
                ->where('tenant_id', $tenantId)
                ->where('user_id', $user->id) // Assuming user_id is the correct column
                ->whereDate('created_at', $today) // Assuming created_at is the correct column
                ->count();
        }
        
        return Inertia::render('Dashboard', [
            'dashboardData' => $dashboardData
        ]);
    }
}
