<?php

namespace App\Http\Controllers\Superadmin;

use App\Http\Controllers\Controller;
use App\Models\Tenant;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use Inertia\Inertia;

class TenantController extends Controller
{
    /**
     * Create a new controller instance.
     *
     * @return void
     */
    public function __construct()
    {
        $this->middleware('permission:access superadmin panel');
    }

    /**
     * Display a listing of the tenants.
     *
     * @return \Inertia\Response
     */
    public function index()
    {
        // Get all tenants
        $tenants = Tenant::latest()->get();
        
        // For each tenant, manually find its admin user
        foreach ($tenants as $tenant) {
            // First try to find users with role='admin' in the users table
            $adminUser = User::where('tenant_id', $tenant->id)
                ->where('role', 'admin')
                ->first();
            
            // If not found, try to find users with the admin-tenant-X role
            if (!$adminUser) {
                $adminRoleName = 'admin-tenant-' . $tenant->id;
                $adminUser = User::where('tenant_id', $tenant->id)
                    ->whereHas('roles', function($query) use ($adminRoleName) {
                        $query->where('name', $adminRoleName);
                    })
                    ->first();
            }
            
            // Add the admin user to the tenant
            if ($adminUser) {
                // We need to manually set the admin user since the relationship might not find it
                $tenant->adminUser = $adminUser;
            }
        }
        
        // Log for debugging
        \Illuminate\Support\Facades\Log::info('Tenants with admin users:', [
            'tenants' => $tenants->map(function($tenant) {
                return [
                    'id' => $tenant->id,
                    'name' => $tenant->name,
                    'admin_user' => $tenant->adminUser ? [
                        'id' => $tenant->adminUser->id,
                        'name' => $tenant->adminUser->name,
                        'email' => $tenant->adminUser->email,
                        'role' => $tenant->adminUser->role,
                    ] : null
                ];
            })
        ]);

        // Transform the tenants collection to include the adminUser property
        $transformedTenants = $tenants->map(function ($tenant) {
            return [
                'id' => $tenant->id,
                'name' => $tenant->name,
                'domain' => $tenant->domain,
                'created_at' => $tenant->created_at,
                'adminUser' => $tenant->adminUser ? [
                    'id' => $tenant->adminUser->id,
                    'name' => $tenant->adminUser->name,
                    'email' => $tenant->adminUser->email,
                    'role' => $tenant->adminUser->role,
                ] : null
            ];
        });
        
        return Inertia::render('Superadmin/Tenants/Index', [
            'tenants' => $transformedTenants
        ]);
    }

    /**
     * Show the form for creating a new tenant.
     *
     * @return \Inertia\Response
     */
    public function create()
    {
        return Inertia::render('Superadmin/Tenants/Create');
    }

    /**
     * Store a newly created tenant in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function store(Request $request)
    {
        // Validate the request data
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'domain' => 'nullable|string|max:255|unique:tenants',
            'admin_name' => 'required|string|max:255',
            'admin_email' => 'required|string|email|max:255|unique:users,email',
            'admin_password' => 'required|string|min:8',
        ]);

        // Log the validation success
        \Illuminate\Support\Facades\Log::info('Tenant creation validation passed', [
            'name' => $validated['name'],
            'domain' => $validated['domain'],
            'admin_name' => $validated['admin_name'],
            'admin_email' => $validated['admin_email'],
        ]);

        try {
            $tenant = DB::transaction(function () use ($validated) {
                // Step 1: Create the tenant
                \Illuminate\Support\Facades\Log::info('Creating tenant', ['name' => $validated['name']]);
                $tenant = Tenant::create([
                    'name' => $validated['name'],
                    'domain' => $validated['domain'],
                ]);
                \Illuminate\Support\Facades\Log::info('Tenant created', ['id' => $tenant->id]);

                // Step 2: Create the admin user
                \Illuminate\Support\Facades\Log::info('Creating admin user', ['email' => $validated['admin_email']]);
                $adminUser = User::create([
                    'name' => $validated['admin_name'],
                    'email' => $validated['admin_email'],
                    'password' => Hash::make($validated['admin_password']),
                    'tenant_id' => $tenant->id,
                    'role' => 'admin', // Set the role field in the users table
                    'is_superadmin' => false,
                ]);
                \Illuminate\Support\Facades\Log::info('Admin user created', ['id' => $adminUser->id]);
                
                // Step 3: Create or get the admin role
                $adminRoleName = 'admin-tenant-' . $tenant->id;
                \Illuminate\Support\Facades\Log::info('Looking for role', ['name' => $adminRoleName]);
                
                // Create the role if it doesn't exist
                $adminRole = \Spatie\Permission\Models\Role::firstOrCreate(
                    ['name' => $adminRoleName],
                    ['guard_name' => 'web']
                );
                \Illuminate\Support\Facades\Log::info('Role found/created', ['id' => $adminRole->id]);
                
                // Step 4: Assign permissions to the role
                \Illuminate\Support\Facades\Log::info('Assigning permissions to role');
                
                // Get all permissions except superadmin ones
                $tenantPermissions = \Spatie\Permission\Models\Permission::whereNotIn('name', [
                    'access superadmin panel', 'access-superadmin-panel', 'manage tenants'
                ])->get();
                
                // Sync permissions to the role
                $adminRole->syncPermissions($tenantPermissions);
                \Illuminate\Support\Facades\Log::info('Permissions assigned', [
                    'count' => $tenantPermissions->count()
                ]);
                
                // Step 5: Assign the role to the user
                \Illuminate\Support\Facades\Log::info('Assigning role to user');
                $adminUser->assignRole($adminRole);
                \Illuminate\Support\Facades\Log::info('Role assigned to user');

                return $tenant;
            });

            \Illuminate\Support\Facades\Log::info('Tenant creation completed successfully', [
                'tenant_id' => $tenant->id,
                'tenant_name' => $tenant->name
            ]);

            return redirect()->route('superadmin.tenants.index')
                ->with('success', 'Tenant created successfully with admin user.');
        } catch (\Exception $e) {
            \Illuminate\Support\Facades\Log::error('Tenant creation failed', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            
            return redirect()->back()
                ->withInput()
                ->with('error', 'Failed to create tenant: ' . $e->getMessage());
        }
    }

    /**
     * Show the form for editing the specified tenant.
     *
     * @param  \App\Models\Tenant  $tenant
     * @return \Inertia\Response
     */
    public function edit(Tenant $tenant)
    {
        // Get the admin user for this tenant
        $adminUser = User::where('tenant_id', $tenant->id)
            ->where('role', 'admin')
            ->first();

        return Inertia::render('Superadmin/Tenants/Edit', [
            'tenant' => $tenant,
            'adminUser' => $adminUser
        ]);
    }

    /**
     * Update the specified tenant in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Models\Tenant  $tenant
     * @return \Illuminate\Http\RedirectResponse
     */
    public function update(Request $request, Tenant $tenant)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'domain' => 'nullable|string|max:255|unique:tenants,domain,' . $tenant->id,
        ]);

        try {
            $tenant->update([
                'name' => $request->name,
                'domain' => $request->domain,
            ]);

            return redirect()->route('superadmin.tenants.index')
                ->with('success', 'Tenant updated successfully.');
        } catch (\Exception $e) {
            return redirect()->back()
                ->withInput()
                ->with('error', 'Failed to update tenant: ' . $e->getMessage());
        }
    }

    /**
     * Remove the specified tenant from storage.
     *
     * @param  \App\Models\Tenant  $tenant
     * @return \Illuminate\Http\RedirectResponse
     */
    public function destroy(Tenant $tenant)
    {
        try {
            DB::transaction(function () use ($tenant) {
                // Delete associated users first
                User::where('tenant_id', $tenant->id)->delete();
                
                // Then delete the tenant
                $tenant->delete();
            });

            return redirect()->route('superadmin.tenants.index')
                ->with('success', 'Tenant and associated users deleted successfully.');
        } catch (\Exception $e) {
            return redirect()->back()
                ->with('error', 'Failed to delete tenant: ' . $e->getMessage());
        }
    }
}
