<?php

namespace App\Http\Requests;

use App\Models\Purchase;
use App\Models\PurchaseItem;
use Illuminate\Foundation\Http\FormRequest;
use Illuminate\Support\Facades\Auth;
use Illuminate\Validation\Rule;

class StorePurchaseReturnRequest extends FormRequest
{
    /**
     * Determine if the user is authorized to make this request.
     */
    public function authorize(): bool
    {
        return $this->user()->can('process-purchase-returns');
    }

    /**
     * Get the validation rules that apply to the request.
     *
     * @return array<string, \Illuminate\Contracts\Validation\ValidationRule|array<mixed>|string>
     */
    public function rules(): array
    {
        $purchaseId = $this->route('purchase');
        $purchase = Purchase::where('id', $purchaseId)
            ->where('tenant_id', Auth::user()->tenant_id)
            ->first();
            
        if (!$purchase) {
            return [];
        }
        
        return [
            'reason' => 'nullable|string',
            'items' => 'required|array|min:1',
            'items.*.purchase_item_id' => [
                'required',
                Rule::exists('purchase_items', 'id')
                    ->where('purchase_id', $purchase->id)
                    ->where('tenant_id', Auth::user()->tenant_id),
            ],
            'items.*.quantity' => [
                'required',
                'integer',
                'min:1',
                function ($attribute, $value, $fail) {
                    $index = explode('.', $attribute)[1];
                    $purchaseItemId = $this->input("items.{$index}.purchase_item_id");
                    
                    if (!$purchaseItemId) {
                        return;
                    }
                    
                    $purchaseItem = PurchaseItem::find($purchaseItemId);
                    if (!$purchaseItem) {
                        return;
                    }
                    
                    // Calculate already returned quantity for this purchase item
                    $alreadyReturnedQty = $purchaseItem->returnItems()->sum('quantity');
                    $maxReturnableQty = $purchaseItem->quantity - $alreadyReturnedQty;
                    
                    if ($value > $maxReturnableQty) {
                        $fail("Cannot return more than {$maxReturnableQty} units for this item.");
                    }
                },
            ],
        ];
    }
}
