<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use Spatie\Permission\Models\Role;
use Spatie\Permission\Models\Permission;
use App\Models\User;

class StandardizePermissions extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'permissions:standardize';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Standardize permissions to use hyphenated format and ensure all admin users have the correct permissions';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $this->info('Standardizing permissions to use hyphenated format...');
        
        // Define the mapping of permissions we want to standardize
        $permissionMapping = [
            'process sales returns' => 'process-sales-returns',
            'process purchase returns' => 'process-purchase-returns',
            'manage sales' => 'manage-sales',
            'manage purchases' => 'manage-purchases',
            'manage clients' => 'manage-clients',
            'manage suppliers' => 'manage-suppliers',
            'manage products' => 'manage-products',
            'manage expenses' => 'manage-expenses',
            'manage users' => 'manage-users',
            'record payments' => 'record-payments',
            'process sales' => 'process-sales',
            'view reports' => 'view-reports',
        ];
        
        // Ensure all hyphenated permissions exist
        foreach ($permissionMapping as $oldFormat => $newFormat) {
            Permission::firstOrCreate(['name' => $newFormat]);
            $this->line("Ensured permission exists: {$newFormat}");
        }
        
        // Update all admin roles to have the hyphenated permissions
        $adminRoles = Role::where('name', 'like', 'admin%')->get();
        $this->info("Found {$adminRoles->count()} admin roles to update.");
        
        foreach ($adminRoles as $role) {
            $this->line("Updating role: {$role->name}");
            
            // Get current permissions
            $currentPermissions = $role->permissions->pluck('name')->toArray();
            
            // Add hyphenated permissions
            foreach ($permissionMapping as $oldFormat => $newFormat) {
                if (in_array($oldFormat, $currentPermissions) && !in_array($newFormat, $currentPermissions)) {
                    $role->givePermissionTo($newFormat);
                    $this->line("  Added {$newFormat} to role {$role->name}");
                }
            }
        }
        
        // Update all users to have the hyphenated permissions if they have the old format
        $users = User::all();
        $this->info("Checking {$users->count()} users for permission updates.");
        
        foreach ($users as $user) {
            $userPermissions = $user->getDirectPermissions()->pluck('name')->toArray();
            
            foreach ($permissionMapping as $oldFormat => $newFormat) {
                if (in_array($oldFormat, $userPermissions) && !in_array($newFormat, $userPermissions)) {
                    $user->givePermissionTo($newFormat);
                    $this->line("Added {$newFormat} to user {$user->name} (ID: {$user->id})");
                }
            }
        }
        
        $this->info('Permission standardization completed successfully!');
        
        return 0;
    }
}
