<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Sale;
use App\Models\SaleItem;
use App\Models\SaleReturn;
use App\Models\SaleReturnItem;
use App\Models\Product;
use App\Models\Client;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Validation\ValidationException;

class SaleReturnController extends Controller
{
    /**
     * Process a sale return
     *
     * @param Request $request
     * @param Sale $sale
     * @return \Illuminate\Http\JsonResponse
     */
    public function store(Request $request, Sale $sale)
    {
        // Check if user has permission to process returns
        if (!Auth::user()->can('process-sales-returns')) {
            return response()->json(['message' => 'Unauthorized'], 403);
        }

        // Validate request
        $validated = $request->validate([
            'reason' => 'required|string|max:255',
            'items' => 'required|array|min:1',
            'items.*.product_id' => 'required|exists:products,id',
            'items.*.quantity' => 'required|integer|min:1',
        ]);

        // Check if the sale belongs to the current tenant
        if ($sale->tenant_id !== Auth::user()->tenant_id) {
            return response()->json(['message' => 'Sale not found'], 404);
        }

        // Check if sale is completed (can't return cancelled or pending sales)
        if ($sale->status !== 'completed') {
            return response()->json([
                'message' => 'Only completed sales can be returned'
            ], 422);
        }

        // Start a database transaction
        DB::beginTransaction();

        try {
            // Create the sale return
            $saleReturn = new SaleReturn([
                'sale_id' => $sale->id,
                'client_id' => $sale->client_id,
                'reason' => $validated['reason'],
                'tenant_id' => Auth::user()->tenant_id,
                'user_id' => Auth::user()->id,
                'returned_amount' => 0, // Will be calculated below
                'status' => 'completed',
            ]);
            $saleReturn->save();

            $totalReturnAmount = 0;

            // Process each return item
            foreach ($validated['items'] as $item) {
                // Find the original sale item
                $saleItem = SaleItem::where('sale_id', $sale->id)
                    ->where('product_id', $item['product_id'])
                    ->first();

                if (!$saleItem) {
                    throw ValidationException::withMessages([
                        'items' => ['Product not found in this sale']
                    ]);
                }

                // Check if return quantity is valid
                if ($item['quantity'] > $saleItem->quantity) {
                    throw ValidationException::withMessages([
                        'items' => ["Cannot return more than purchased quantity for product: {$saleItem->product->name}"]
                    ]);
                }

                // Calculate the return amount for this item
                $returnAmount = $item['quantity'] * $saleItem->price_at_sale;
                $totalReturnAmount += $returnAmount;

                // Create the return item
                $returnItem = new SaleReturnItem([
                    'tenant_id' => Auth::user()->tenant_id,
                    'sale_return_id' => $saleReturn->id,
                    'product_id' => $item['product_id'],
                    'sale_item_id' => $saleItem->id,
                    'quantity' => $item['quantity'],
                    'price_at_sale' => $saleItem->price_at_sale,
                    'subtotal' => $returnAmount,
                ]);
                $returnItem->save();

                // Update product inventory
                $product = Product::find($item['product_id']);
                $product->quantity += $item['quantity'];
                $product->save();
            }

            // Update the returned amount on the return
            $saleReturn->returned_amount = $totalReturnAmount;
            $saleReturn->save();

            // Update client balance (reduce what they owe)
            $client = Client::find($sale->client_id);
            $client->lockForUpdate();
            $client->balance -= $totalReturnAmount;
            $client->save();

            DB::commit();

            return response()->json([
                'message' => 'Sale return processed successfully',
                'sale_return' => $saleReturn->load('items.product')
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'message' => 'An error occurred while processing the return',
                'error' => $e->getMessage()
            ], 500);
        }
    }
}
