<?php

namespace App\Http\Controllers;

use App\Http\Requests\StoreSaleRequest;
use App\Models\Client;
use App\Models\Product;
use App\Models\Sale;
use App\Models\SaleItem;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class SaleController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        // Get sales for the current tenant with pagination and eager load relationships
        // Only load the necessary relationships for the index view to improve performance
        $sales = Sale::with(['user:id,name', 'client:id,name'])
            ->where('tenant_id', Auth::user()->tenant_id)
            ->orderBy('created_at', 'desc')
            ->paginate(15);
            
        return response()->json($sales);
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(StoreSaleRequest $request)
    {
        // Use a database transaction to ensure atomicity
        return DB::transaction(function () use ($request) {
            // Dump the entire request for debugging
            \Log::info('Sale request data: ' . json_encode($request->all()));
            
            $validatedData = $request->validated();
            $items = $validatedData['items'];
            $status = $request->input('status', 'completed');
            
            // Get the client
            $client = Client::where('id', $request->client_id)
                ->where('tenant_id', Auth::user()->tenant_id)
                ->lockForUpdate() // Lock for update to prevent race conditions
                ->first();
                
            if (!$client) {
                return response()->json(['message' => 'Client not found'], 404);
            }
            
            // Create the sale record first (without total amount)
            $sale = new Sale();
            $sale->tenant_id = Auth::user()->tenant_id;
            $sale->user_id = Auth::id();
            $sale->client_id = $client->id;
            $sale->status = $status;
            $sale->save();
            
            // Track the total amount
            $totalAmount = 0;
            
            // Process each item
            foreach ($items as $item) {
                // Get the product
                $product = Product::where('id', $item['product_id'])
                    ->where('tenant_id', Auth::user()->tenant_id)
                    ->lockForUpdate() // Lock to prevent race conditions
                    ->first();
                
                if (!$product) {
                    return response()->json(['message' => 'Product not found: ' . $item['product_id']], 404);
                }
                
                // Check if we have enough stock
                if ($product->quantity < $item['quantity']) {
                    return response()->json([
                        'message' => "Insufficient stock for product: {$product->name}",
                        'available' => $product->quantity,
                        'requested' => $item['quantity']
                    ], 422);
                }
                
                // Determine the price to use
                $priceToUse = isset($item['selling_price']) ? $item['selling_price'] : $product->price;
                \Log::info("Item {$product->id} price: {$priceToUse}");
                
                // Create the sale item
                $saleItem = new SaleItem();
                $saleItem->tenant_id = Auth::user()->tenant_id;
                $saleItem->sale_id = $sale->id;
                $saleItem->product_id = $product->id;
                $saleItem->quantity = $item['quantity'];
                $saleItem->price_at_sale = $priceToUse;
                $saleItem->save();
                
                // Update product quantity
                $product->quantity -= $item['quantity'];
                $product->save();
                
                // Add to total
                $itemTotal = (float)$priceToUse * (int)$item['quantity'];
                \Log::info("Item total: {$itemTotal} = {$priceToUse} * {$item['quantity']}");
                $totalAmount += $itemTotal;
            }
            
            // Round the total amount
            $totalAmount = round($totalAmount, 2);
            \Log::info("Final total amount: {$totalAmount}");
            
            // Update the sale with the total amount
            $sale->total_amount = $totalAmount;
            $sale->save();
            
            // Verify the total was saved correctly
            $dbValue = \DB::table('sales')->where('id', $sale->id)->value('total_amount');
            \Log::info("Database value for sale {$sale->id}: {$dbValue}");
            
            // Force update if needed
            if ((float)$dbValue != (float)$totalAmount) {
                \Log::info("Mismatch detected! Forcing update via direct query.");
                \DB::statement("UPDATE sales SET total_amount = ? WHERE id = ?", [$totalAmount, $sale->id]);
            }
            
            // This section is never executed because we're already creating sale items above
            // Removing this duplicate code
            
            // Update client balance (client owes more)
            $client->balance += $totalAmount;
            $client->save();
            
            // The creation of sale items has been moved above
            
            // Load relationships for the response
            $sale->load(['user', 'client', 'saleItems.product']);
            
            return response()->json($sale, 201);
        });
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        // Find the sale belonging to the current tenant
        $sale = Sale::with(['user', 'saleItems.product'])
            ->where('id', $id)
            ->where('tenant_id', Auth::user()->tenant_id)
            ->first();
            
        if (!$sale) {
            return response()->json(['message' => 'Sale not found'], 404);
        }
        
        return response()->json($sale);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id)
    {
        // Find the sale belonging to the current tenant
        $sale = Sale::where('id', $id)
            ->where('tenant_id', Auth::user()->tenant_id)
            ->first();
            
        if (!$sale) {
            return response()->json(['message' => 'Sale not found'], 404);
        }
        
        // Only allow updating the status
        $request->validate([
            'status' => 'required|string|in:completed,pending,cancelled',
        ]);
        
        $sale->status = $request->input('status');
        $sale->save();
        
        return response()->json($sale);
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        // Find the sale belonging to the current tenant
        $sale = Sale::where('id', $id)
            ->where('tenant_id', Auth::user()->tenant_id)
            ->first();
            
        if (!$sale) {
            return response()->json(['message' => 'Sale not found'], 404);
        }
        
        // Only allow deleting pending sales
        if ($sale->status !== 'pending') {
            return response()->json(['message' => 'Only pending sales can be deleted'], 422);
        }
        
        // Use a transaction to ensure atomicity
        DB::transaction(function () use ($sale) {
            // Get the client and lock for update to prevent race conditions
            $client = Client::where('id', $sale->client_id)
                ->where('tenant_id', Auth::user()->tenant_id)
                ->lockForUpdate()
                ->first();
                
            if (!$client) {
                throw new \Exception('Client not found');
            }
            
            // Update client balance (client owes less)
            $client->balance -= $sale->total_amount;
            $client->save();
            
            // Restore product quantities
            foreach ($sale->saleItems as $item) {
                $product = $item->product;
                $product->quantity += $item->quantity;
                $product->save();
            }
            
            // Delete the sale (this will cascade to sale items due to foreign key constraint)
            $sale->delete();
        });
        
        return response()->json(null, 204);
    }
}
