<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Purchase;
use App\Models\PurchaseItem;
use App\Models\PurchaseReturn;
use App\Models\PurchaseReturnItem;
use App\Models\Product;
use App\Models\Supplier;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Validation\ValidationException;

class PurchaseReturnController extends Controller
{
    /**
     * Process a purchase return
     *
     * @param Request $request
     * @param Purchase $purchase
     * @return \Illuminate\Http\JsonResponse
     */
    public function store(Request $request, Purchase $purchase)
    {
        // Check if user has permission to process returns
        if (!Auth::user()->can('process-purchase-returns')) {
            return response()->json(['message' => 'Unauthorized'], 403);
        }

        // Validate request
        $validated = $request->validate([
            'reason' => 'required|string|max:255',
            'items' => 'required|array|min:1',
            'items.*.product_id' => 'required|exists:products,id',
            'items.*.purchase_item_id' => 'required|exists:purchase_items,id',
            'items.*.quantity' => 'required|integer|min:1',
        ]);

        // Check if the purchase belongs to the current tenant
        if ($purchase->tenant_id !== Auth::user()->tenant_id) {
            return response()->json(['message' => 'Purchase not found'], 404);
        }

        // Check if purchase is completed (can't return cancelled or pending purchases)
        if ($purchase->status !== 'completed') {
            return response()->json([
                'message' => 'Only completed purchases can be returned'
            ], 422);
        }

        // Start a database transaction
        DB::beginTransaction();

        try {
            // Create the purchase return
            $purchaseReturn = new PurchaseReturn([
                'purchase_id' => $purchase->id,
                'supplier_id' => $purchase->supplier_id,
                'reason' => $validated['reason'],
                'tenant_id' => Auth::user()->tenant_id,
                'user_id' => Auth::user()->id,
                'returned_amount' => 0, // Will be calculated below
                'status' => 'completed',
            ]);
            $purchaseReturn->save();

            $totalReturnAmount = 0;

            // Process each return item
            foreach ($validated['items'] as $item) {
                // Find the original purchase item
                $purchaseItem = PurchaseItem::where('id', $item['purchase_item_id'])
                    ->where('purchase_id', $purchase->id)
                    ->where('product_id', $item['product_id'])
                    ->first();

                if (!$purchaseItem) {
                    throw ValidationException::withMessages([
                        'items' => ['Purchase item not found']
                    ]);
                }

                // Check if return quantity is valid
                if ($item['quantity'] > $purchaseItem->quantity) {
                    throw ValidationException::withMessages([
                        'items' => ["Cannot return more than purchased quantity for product: {$purchaseItem->product->name}"]
                    ]);
                }

                // Calculate the return amount for this item
                $returnAmount = $item['quantity'] * $purchaseItem->cost_price;
                $totalReturnAmount += $returnAmount;

                // Create the return item
                $returnItem = new PurchaseReturnItem([
                    'tenant_id' => Auth::user()->tenant_id,
                    'purchase_return_id' => $purchaseReturn->id,
                    'product_id' => $item['product_id'],
                    'purchase_item_id' => $purchaseItem->id,
                    'quantity' => $item['quantity'],
                    'cost_price' => $purchaseItem->cost_price,
                    'subtotal' => $returnAmount,
                ]);
                $returnItem->save();

                // Update product inventory
                $product = Product::find($item['product_id']);
                $product->quantity -= $item['quantity'];
                $product->save();
            }

            // Update the returned amount on the return
            $purchaseReturn->returned_amount = $totalReturnAmount;
            $purchaseReturn->save();

            // Update supplier balance (reduce what we owe)
            $supplier = Supplier::find($purchase->supplier_id);
            $supplier->lockForUpdate();
            $supplier->balance -= $totalReturnAmount;
            $supplier->save();

            DB::commit();

            return response()->json([
                'message' => 'Purchase return processed successfully',
                'purchase_return' => $purchaseReturn->load('items.product')
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'message' => 'An error occurred while processing the return',
                'error' => $e->getMessage()
            ], 500);
        }
    }
}
