<?php

namespace App\Traits;

use Illuminate\Database\Eloquent\MissingAttributeException;

trait PreventMissingAttributes
{
    /**
     * Throw an exception when trying to access a missing attribute.
     *
     * @param  string  $key
     * @return mixed
     *
     * @throws \Illuminate\Database\Eloquent\MissingAttributeException
     */
    public function getAttributeValue($key)
    {
        if (! $this->isAttributeRetrievable($key)) {
            throw new MissingAttributeException($this, $key);
        }

        return parent::getAttributeValue($key);
    }

    /**
     * Determine if the given attribute is retrievable.
     *
     * @param  string  $key
     * @return bool
     */
    protected function isAttributeRetrievable($key)
    {
        return array_key_exists($key, $this->attributes) ||
               $this->hasGetMutator($key) ||
               $this->hasAttributeMutator($key) ||
               $this->isClassCastable($key) ||
               in_array($key, $this->getDates()) ||
               $this->hasColumn($key);
    }

    /**
     * Determine if the model has a column with the given key.
     *
     * @param  string  $key
     * @return bool
     */
    protected function hasColumn($key)
    {
        static $cache = [];

        $class = get_class($this);

        if (! isset($cache[$class])) {
            $cache[$class] = $this->getConnection()
                ->getSchemaBuilder()
                ->hasColumn($this->getTable(), $key);
        }

        return $cache[$class];
    }
}
