<?php

namespace App\Helpers;

/**
 * Helper class for permission standardization
 * This class provides a consistent way to map between different permission formats
 * while maintaining backward compatibility during the transition.
 */
class PermissionHelper
{
    /**
     * Map of space-separated permissions to hyphenated format
     */
    public static $permissionMap = [
        'process sales returns' => 'process-sales-returns',
        'process purchase returns' => 'process-purchase-returns',
        'manage sales' => 'manage-sales',
        'manage purchases' => 'manage-purchases',
        'manage clients' => 'manage-clients',
        'manage suppliers' => 'manage-suppliers',
        'manage products' => 'manage-products',
        'manage expenses' => 'manage-expenses',
        'manage users' => 'manage-users',
        'record payments' => 'record-payments',
        'process sales' => 'process-sales',
        'view reports' => 'view-reports',
        'access superadmin panel' => 'access-superadmin-panel',
    ];

    /**
     * Get the standardized (hyphenated) permission name
     *
     * @param string $permission
     * @return string
     */
    public static function getStandardized($permission)
    {
        return self::$permissionMap[$permission] ?? $permission;
    }

    /**
     * Get all equivalent permission names (both formats)
     *
     * @param string $permission
     * @return array
     */
    public static function getAllEquivalents($permission)
    {
        // If it's a space-separated permission, return both formats
        if (isset(self::$permissionMap[$permission])) {
            return [$permission, self::$permissionMap[$permission]];
        }
        
        // If it's a hyphenated permission, find its space-separated equivalent
        $spaceFormat = array_search($permission, self::$permissionMap);
        if ($spaceFormat !== false) {
            return [$spaceFormat, $permission];
        }
        
        // If it's not in our map, just return the original
        return [$permission];
    }
}
