<?php

namespace App\Traits;

use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;

trait PermissionCheckTrait
{
    /**
     * Check if the user has any of the specified permissions or is an admin
     *
     * @param array $permissions Array of permission names to check
     * @return bool
     */
    protected function userHasAnyPermission(array $permissions)
    {
        $user = Auth::user();
        
        // Superadmins have all permissions
        if ($user->is_superadmin) {
            Log::info('User is superadmin, granting all permissions');
            return true;
        }
        
        // Check if the user has an admin role through Spatie role system
        $hasAdminRole = false;
        $userRoles = $user->getRoleNames()->toArray();
        foreach ($userRoles as $roleName) {
            if (strpos($roleName, 'admin-tenant-') === 0) {
                $hasAdminRole = true;
                break;
            }
        }
        
        // Admin users have all tenant permissions except superadmin
        if ($user->role === 'admin' || $hasAdminRole) {
            // Check if any of the requested permissions are superadmin-only
            $superadminPermissions = ['access superadmin panel', 'manage tenants'];
            $requestingSuperadminPermission = false;
            
            foreach ($permissions as $permission) {
                if (in_array($permission, $superadminPermissions)) {
                    $requestingSuperadminPermission = true;
                    break;
                }
            }
            
            // If not requesting superadmin permissions, grant access to admin
            if (!$requestingSuperadminPermission) {
                Log::info('User is admin, granting tenant permissions');
                return true;
            }
        }
        
        // Get all user permissions for direct checking
        $userPermissions = $user->getAllPermissions()->pluck('name')->toArray();
        
        // Log for debugging
        Log::info('Checking permissions for user: ' . $user->id, [
            'user_permissions' => $userPermissions,
            'checking_for' => $permissions
        ]);
        
        // Generate all possible variations of each permission
        $permissionVariations = $this->generatePermissionVariations($permissions);
        
        // Check if user has any of the permission variations
        foreach ($permissionVariations as $permVariation) {
            if (in_array($permVariation, $userPermissions)) {
                Log::info('Permission match found: ' . $permVariation);
                return true;
            }
        }
        
        return false;
    }
    
    /**
     * Generate all possible variations of the given permissions
     * This includes:
     * - Original format
     * - With spaces instead of hyphens
     * - With hyphens instead of spaces
     * - Singular form
     * - Plural form
     *
     * @param array $permissions
     * @return array
     */
    protected function generatePermissionVariations(array $permissions)
    {
        $variations = [];
        
        foreach ($permissions as $permission) {
            // Add original permission
            $variations[] = $permission;
            
            // Add with spaces instead of hyphens
            $withSpaces = str_replace('-', ' ', $permission);
            if ($withSpaces !== $permission) {
                $variations[] = $withSpaces;
            }
            
            // Add with hyphens instead of spaces
            $withHyphens = str_replace(' ', '-', $permission);
            if ($withHyphens !== $permission) {
                $variations[] = $withHyphens;
            }
            
            // Handle singular/plural variations
            // Extract the action and resource parts (e.g., 'view' and 'products')
            if (preg_match('/^(\w+)\s+(.+)$/', $permission, $matches)) {
                $action = $matches[1]; // e.g., 'view'
                $resource = $matches[2]; // e.g., 'products'
                
                // Try singular form (remove trailing 's')
                if (substr($resource, -1) === 's') {
                    $singularResource = substr($resource, 0, -1);
                    $variations[] = $action . ' ' . $singularResource;
                    $variations[] = $action . '-' . $singularResource;
                }
                
                // Try plural form (add trailing 's')
                if (substr($resource, -1) !== 's') {
                    $pluralResource = $resource . 's';
                    $variations[] = $action . ' ' . $pluralResource;
                    $variations[] = $action . '-' . $pluralResource;
                }
            }
            
            // Handle hyphenated versions of the above
            if (preg_match('/^(\w+)\-(.+)$/', $permission, $matches)) {
                $action = $matches[1]; // e.g., 'view'
                $resource = $matches[2]; // e.g., 'products'
                
                // Try singular form (remove trailing 's')
                if (substr($resource, -1) === 's') {
                    $singularResource = substr($resource, 0, -1);
                    $variations[] = $action . ' ' . $singularResource;
                    $variations[] = $action . '-' . $singularResource;
                }
                
                // Try plural form (add trailing 's')
                if (substr($resource, -1) !== 's') {
                    $pluralResource = $resource . 's';
                    $variations[] = $action . ' ' . $pluralResource;
                    $variations[] = $action . '-' . $pluralResource;
                }
            }
        }
        
        return array_unique($variations);
    }
    
    /**
     * Check if the user has any of the specified permissions or is an admin
     * If not, abort with a 403 error
     *
     * @param array $permissions Array of permission names to check
     * @return void
     */
    protected function authorizeAnyPermission(array $permissions)
    {
        $user = Auth::user();
        $userPermissions = $user->getAllPermissions()->pluck('name')->toArray();
        
        if (!$this->userHasAnyPermission($permissions)) {
            Log::warning('Permission denied for user: ' . $user->id, [
                'user_permissions' => $userPermissions,
                'required_permissions' => $permissions
            ]);
            abort(403, 'User does not have the right permissions.');
        }
    }
}
