<?php

namespace App\Http\Middleware;

use Illuminate\Http\Request;
use Inertia\Middleware;

class HandleInertiaRequests extends Middleware
{
    /**
     * The root template that's loaded on the first page visit.
     *
     * @see https://inertiajs.com/server-side-setup#root-template
     * @var string
     */
    protected $rootView = 'app';

    /**
     * Determines the current asset version.
     *
     * @see https://inertiajs.com/asset-versioning
     * @param  \Illuminate\Http\Request  $request
     * @return string|null
     */
    public function version(Request $request): ?string
    {
        return parent::version($request);
    }

    /**
     * Defines the props that are shared by default.
     *
     * @see https://inertiajs.com/shared-data
     * @param  \Illuminate\Http\Request  $request
     * @return array
     */
    public function share(Request $request): array
    {
        return array_merge(parent::share($request), [
            'auth' => function () use ($request) {
                $user = $request->user();
                return [
                    'user' => $user ? [
                        'id' => $user->id,
                        'name' => $user->name,
                        'email' => $user->email,
                        'role' => $user->role,
                        'tenant_id' => $user->tenant_id,
                        'is_superadmin' => $user->is_superadmin,
                        'permissions' => $user ? $this->getUserPermissions($user) : [],
                    ] : null,
                ];
            },
        ]);
    }
    
    /**
     * Get the permissions for the user.
     *
     * @param  \App\Models\User  $user
     * @return array
     */
    protected function getUserPermissions($user): array
    {
        // Use Spatie's permission system to get all permissions for the user
        if (!$user) {
            return [];
        }
        
        // Superadmins and tenant admins get all permissions
        if ($user->is_superadmin || $user->role === 'admin') {
            // Get all available permissions in the system
            return \Spatie\Permission\Models\Permission::all()->pluck('name')->toArray();
        }
        
        // For other users, get permissions through roles
        return $user->getAllPermissions()->pluck('name')->toArray();
    }
}
