<?php

namespace App\Http\Controllers;

use App\Http\Requests\StoreExpenseRequest;
use App\Http\Requests\UpdateExpenseRequest;
use App\Models\Expense;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class ExpenseController extends Controller
{
    /**
     * Create a new controller instance.
     */
    public function __construct()
    {
        $this->middleware('permission:manage expenses');
    }
    
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $query = Expense::where('tenant_id', Auth::user()->tenant_id);
        
        // Filter by category if provided
        if ($request->has('category')) {
            $query->where('category', $request->category);
        }
        
        // Filter by date range if provided
        if ($request->has('start_date')) {
            $query->where('expense_date', '>=', $request->start_date);
        }
        
        if ($request->has('end_date')) {
            $query->where('expense_date', '<=', $request->end_date);
        }
        
        $expenses = $query->with('user:id,name')
            ->orderBy('expense_date', 'desc')
            ->paginate(15);
            
        return response()->json($expenses);
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(StoreExpenseRequest $request)
    {
        $expense = new Expense([
            'tenant_id' => Auth::user()->tenant_id,
            'user_id' => Auth::id(),
            'category' => $request->category,
            'description' => $request->description,
            'amount' => $request->amount,
            'expense_date' => $request->expense_date,
        ]);
        
        $expense->save();
        
        return response()->json($expense, 201);
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        $expense = Expense::where('id', $id)
            ->where('tenant_id', Auth::user()->tenant_id)
            ->with('user:id,name')
            ->first();
            
        if (!$expense) {
            return response()->json(['message' => 'Expense not found'], 404);
        }
        
        return response()->json($expense);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(UpdateExpenseRequest $request, string $id)
    {
        $expense = Expense::where('id', $id)
            ->where('tenant_id', Auth::user()->tenant_id)
            ->first();
            
        if (!$expense) {
            return response()->json(['message' => 'Expense not found'], 404);
        }
        
        if ($request->filled('category')) $expense->category = $request->category;
        if ($request->filled('description')) $expense->description = $request->description;
        if ($request->filled('amount')) $expense->amount = $request->amount;
        if ($request->filled('expense_date')) $expense->expense_date = $request->expense_date;
        
        $expense->save();
        
        return response()->json($expense);
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        $expense = Expense::where('id', $id)
            ->where('tenant_id', Auth::user()->tenant_id)
            ->first();
            
        if (!$expense) {
            return response()->json(['message' => 'Expense not found'], 404);
        }
        
        $expense->delete();
        
        return response()->json(null, 204);
    }
}
