<?php

namespace App\Http\Controllers;

use App\Models\Client;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class ClientController extends Controller
{
    /**
     * Create a new controller instance.
     */
    public function __construct()
    {
        $this->middleware('permission:manage clients')->except(['index', 'show']);
    }
    
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $clients = Client::where('tenant_id', Auth::user()->tenant_id)
            ->orderBy('name')
            ->paginate(15);
            
        return response()->json($clients);
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'nullable|email|max:255|unique:clients,email,NULL,id,tenant_id,' . Auth::user()->tenant_id,
            'phone' => 'nullable|string|max:20',
            'address' => 'nullable|string|max:255',
            'notes' => 'nullable|string',
            'is_default' => 'boolean',
        ]);
        
        // If this client is set as default, unset any existing default client
        if ($request->is_default) {
            Client::where('tenant_id', Auth::user()->tenant_id)
                ->where('is_default', true)
                ->update(['is_default' => false]);
        }
        
        $client = new Client([
            'tenant_id' => Auth::user()->tenant_id,
            'name' => $request->name,
            'email' => $request->email,
            'phone' => $request->phone,
            'address' => $request->address,
            'notes' => $request->notes,
            'balance' => 0, // New clients start with zero balance
            'is_default' => $request->is_default ?? false,
        ]);
        
        $client->save();
        
        return response()->json($client, 201);
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        $client = Client::where('id', $id)
            ->where('tenant_id', Auth::user()->tenant_id)
            ->first();
            
        if (!$client) {
            return response()->json(['message' => 'Client not found'], 404);
        }
        
        return response()->json($client);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id)
    {
        $client = Client::where('id', $id)
            ->where('tenant_id', Auth::user()->tenant_id)
            ->first();
            
        if (!$client) {
            return response()->json(['message' => 'Client not found'], 404);
        }
        
        $request->validate([
            'name' => 'sometimes|required|string|max:255',
            'email' => 'nullable|email|max:255|unique:clients,email,' . $client->id . ',id,tenant_id,' . Auth::user()->tenant_id,
            'phone' => 'nullable|string|max:20',
            'address' => 'nullable|string|max:255',
            'notes' => 'nullable|string',
            'is_default' => 'boolean',
        ]);
        
        // If this client is being set as default, unset any existing default client
        if ($request->filled('is_default') && $request->is_default && !$client->is_default) {
            Client::where('tenant_id', Auth::user()->tenant_id)
                ->where('is_default', true)
                ->update(['is_default' => false]);
        }
        
        // Update client fields
        if ($request->filled('name')) $client->name = $request->name;
        if ($request->filled('email')) $client->email = $request->email;
        if ($request->filled('phone')) $client->phone = $request->phone;
        if ($request->filled('address')) $client->address = $request->address;
        if ($request->filled('notes')) $client->notes = $request->notes;
        if ($request->filled('is_default')) $client->is_default = $request->is_default;
        
        $client->save();
        
        return response()->json($client);
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        $client = Client::where('id', $id)
            ->where('tenant_id', Auth::user()->tenant_id)
            ->first();
            
        if (!$client) {
            return response()->json(['message' => 'Client not found'], 404);
        }
        
        // Check if client has any sales
        $salesCount = $client->sales()->count();
        if ($salesCount > 0) {
            return response()->json([
                'message' => 'Cannot delete client with associated sales',
                'sales_count' => $salesCount
            ], 422);
        }
        
        // Check if client has any payments
        $paymentsCount = $client->payments()->count();
        if ($paymentsCount > 0) {
            return response()->json([
                'message' => 'Cannot delete client with associated payments',
                'payments_count' => $paymentsCount
            ], 422);
        }
        
        // Check if client has outstanding balance
        if ($client->balance > 0) {
            return response()->json([
                'message' => 'Cannot delete client with outstanding balance',
                'balance' => $client->balance
            ], 422);
        }
        
        $client->delete();
        
        return response()->json(null, 204);
    }
}
