<?php

namespace App\Providers;

use App\Models\User;
use Illuminate\Support\Facades\Gate;
use Illuminate\Foundation\Support\Providers\AuthServiceProvider as ServiceProvider;
use Spatie\Permission\Models\Permission;
use Spatie\Permission\Models\Role;

class AuthServiceProvider extends ServiceProvider
{
    /**
     * The model to policy mappings for the application.
     *
     * @var array<class-string, class-string>
     */
    protected $policies = [
        //
    ];

    /**
     * Register any authentication / authorization services.
     */
    public function boot(): void
    {
        // Define superadmin access gate - only actual superadmins can access the superadmin panel
        Gate::define('access superadmin panel', fn(User $user) => $user->is_superadmin);
        Gate::define('access-superadmin-panel', fn(User $user) => $user->is_superadmin);
        
        // Define a gate to control who can assign superadmin permissions
        Gate::define('assign-superadmin-permissions', function (User $user) {
            // Only superadmins can assign superadmin permissions
            return $user->is_superadmin;
        });
        
        // Register a gate for checking permissions through Spatie
        Gate::before(function (User $user, $ability) {
            // Superadmins have all abilities automatically
            if ($user->is_superadmin) {
                return true;
            }
            
            // Check if the user has an admin role through Spatie role system
            $hasAdminRole = false;
            $userRoles = $user->getRoleNames()->toArray();
            foreach ($userRoles as $roleName) {
                if (strpos($roleName, 'admin-tenant-') === 0) {
                    $hasAdminRole = true;
                    break;
                }
            }
            
            // Tenant admins have all abilities within their tenant except superadmin features
            if (($user->role === 'admin' || $hasAdminRole) && 
                !in_array($ability, ['access superadmin panel', 'access-superadmin-panel', 'assign-superadmin-permissions', 'manage tenants'])) {
                
                // For non-superadmin abilities, check if it's related to expenses
                // Force strict permission checking for expenses
                if (strpos($ability, 'expense') !== false) {
                    // For expense-related abilities, check actual permissions
                    return null; // Continue to check actual permissions
                }
                
                // For other abilities, grant access to admin
                return true;
            }
            
            // Get all user permissions for direct checking
            $userPermissions = $user->getAllPermissions()->pluck('name')->toArray();
            
            // Check the exact permission
            if (in_array($ability, $userPermissions)) {
                return true;
            }
            
            // Check the normalized version (spaces instead of hyphens)
            $normalizedAbility = str_replace('-', ' ', $ability);
            if ($normalizedAbility !== $ability && in_array($normalizedAbility, $userPermissions)) {
                return true;
            }
            
            // Check the hyphenated version (hyphens instead of spaces)
            $hyphenatedAbility = str_replace(' ', '-', $ability);
            if ($hyphenatedAbility !== $ability && in_array($hyphenatedAbility, $userPermissions)) {
                return true;
            }
        });
    }
}
