<?php

namespace Database\Seeders;

use App\Models\Tenant;
use App\Models\User;
use Illuminate\Database\Seeder;
use Illuminate\Support\Facades\Hash;
use Spatie\Permission\Models\Permission;
use Spatie\Permission\Models\Role;

class UserSeeder extends Seeder
{
    /**
     * Run the database seeds.
     */
    public function run(): void
    {
        // Get all tenants
        $tenants = Tenant::all();
        
        // Assign superadmin role to the superadmin user
        $superadminUser = User::where('is_superadmin', true)->first();
        if ($superadminUser) {
            $superadminUser->assignRole('superadmin');
        }
        
        foreach ($tenants as $tenant) {
            // Create tenant-specific roles with unique names
            $adminRoleName = 'admin-tenant-' . $tenant->id;
            $cashierRoleName = 'cashier-tenant-' . $tenant->id;
            
            $adminRole = Role::create(['name' => $adminRoleName]);
            $cashierRole = Role::create(['name' => $cashierRoleName]);
            
            // Define admin permissions
            $adminPermissions = [
                'view dashboard',
                'view products', 'create products', 'edit products', 'delete products',
                'view sales', 'create sales', 'edit sales', 'delete sales', 'process sales',
                'view clients', 'create clients', 'edit clients', 'delete clients',
                'view suppliers', 'create suppliers', 'edit suppliers', 'delete suppliers',
                'view payments', 'record payments',
                'view purchases', 'create purchases', 'edit purchases', 'delete purchases',
                'view expenses', 'create expenses', 'edit expenses', 'delete expenses',
                'view sales returns', 'process sales returns',
                'view purchase returns', 'process purchase returns',
                'view reports',
                'view users', 'create users', 'edit users', 'delete users',
                'view tenant settings', 'edit tenant settings'
            ];
            
            // Define cashier permissions
            $cashierPermissions = [
                'view dashboard',
                'view products',
                'view sales', 'create sales', 'process sales',
                'view clients', 'create clients',
                'view payments', 'record payments'
            ];
            
            // Assign permissions to roles
            foreach ($adminPermissions as $permission) {
                $adminRole->givePermissionTo($permission);
            }
            
            foreach ($cashierPermissions as $permission) {
                $cashierRole->givePermissionTo($permission);
            }
            
            // Create admin user for each tenant
            $adminUser = User::create([
                'name' => 'Admin User',
                'email' => 'admin@' . strtolower(str_replace(' ', '-', $tenant->name)) . '.com',
                'password' => Hash::make('password'),
                'tenant_id' => $tenant->id,
                'role' => 'admin',
                'is_superadmin' => false,
            ]);
            
            // Assign admin role to admin user
            $adminUser->assignRole($adminRole);
            
            // Create cashier user for each tenant
            $cashierUser = User::create([
                'name' => 'Cashier User',
                'email' => 'cashier@' . strtolower(str_replace(' ', '-', $tenant->name)) . '.com',
                'password' => Hash::make('password'),
                'tenant_id' => $tenant->id,
                'role' => 'cashier',
                'is_superadmin' => false,
            ]);
            
            // Assign cashier role to cashier user
            $cashierUser->assignRole($cashierRole);
        }
    }
}
