<?php

namespace Database\Seeders;

use Illuminate\Database\Seeder;
use Spatie\Permission\Models\Permission;
use Spatie\Permission\Models\Role;
use Spatie\Permission\PermissionRegistrar;
use Illuminate\Support\Facades\DB;

class ConsolidatePermissionsSeeder extends Seeder
{
    /**
     * Run the database seeds.
     */
    public function run(): void
    {
        // Reset cached roles and permissions
        app()[PermissionRegistrar::class]->forgetCachedPermissions();

        // Get all permissions
        $permissions = Permission::all();
        
        // Group permissions by their normalized name (spaces only)
        $permissionGroups = [];
        foreach ($permissions as $permission) {
            $normalizedName = str_replace('-', ' ', $permission->name);
            $permissionGroups[$normalizedName][] = $permission;
        }
        
        // For each group, keep the space version and migrate all role assignments
        foreach ($permissionGroups as $normalizedName => $permissionGroup) {
            if (count($permissionGroup) <= 1) {
                continue; // Skip if there's only one permission with this name
            }
            
            // Find the space version
            $spaceVersion = null;
            $otherVersions = [];
            
            foreach ($permissionGroup as $permission) {
                if ($permission->name === $normalizedName) {
                    $spaceVersion = $permission;
                } else {
                    $otherVersions[] = $permission;
                }
            }
            
            // If no space version exists, create it
            if (!$spaceVersion) {
                $spaceVersion = Permission::create(['name' => $normalizedName]);
                $this->command->info("Created space version: {$normalizedName}");
            }
            
            // For each other version, migrate role assignments and user assignments
            foreach ($otherVersions as $otherVersion) {
                $this->command->info("Migrating from {$otherVersion->name} to {$spaceVersion->name}");
                
                // Get all roles that have this permission
                $roles = Role::with('permissions')->get()
                    ->filter(function ($role) use ($otherVersion) {
                        return $role->permissions->contains('id', $otherVersion->id);
                    });
                
                // Assign the space version to these roles
                foreach ($roles as $role) {
                    if (!$role->hasPermissionTo($spaceVersion)) {
                        $role->givePermissionTo($spaceVersion);
                        $this->command->info("  Assigned {$spaceVersion->name} to role {$role->name}");
                    }
                }
                
                // Get all users that have this permission directly
                $userPermissions = DB::table('model_has_permissions')
                    ->where('permission_id', $otherVersion->id)
                    ->get();
                
                // Assign the space version to these users
                foreach ($userPermissions as $userPermission) {
                    DB::table('model_has_permissions')->updateOrInsert(
                        [
                            'model_type' => $userPermission->model_type,
                            'model_id' => $userPermission->model_id,
                            'permission_id' => $spaceVersion->id
                        ],
                        []
                    );
                    $this->command->info("  Migrated permission for user ID {$userPermission->model_id}");
                }
            }
        }
        
        // Now update the AuthServiceProvider to handle only space versions
        $this->command->info("Permissions consolidated. Remember to update your AuthServiceProvider to handle only space versions.");
    }
}
