<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;
use Illuminate\Support\Facades\DB;
use Spatie\Permission\Models\Permission;
use Spatie\Permission\Models\Role;

return new class extends Migration
{
    /**
     * Run the migrations.
     */
    public function up(): void
    {
        // Step 1: Identify and standardize permission formats
        // We'll standardize on the format with spaces (e.g., "view expenses" instead of "view-expenses")
        
        // Get all permissions
        $permissions = Permission::all();
        
        // Group permissions by their normalized name (without hyphens)
        $permissionGroups = [];
        foreach ($permissions as $permission) {
            $normalizedName = str_replace('-', ' ', $permission->name);
            if (!isset($permissionGroups[$normalizedName])) {
                $permissionGroups[$normalizedName] = [];
            }
            $permissionGroups[$normalizedName][] = $permission;
        }
        
        // Process each group to keep only one permission and update references
        foreach ($permissionGroups as $normalizedName => $permissionGroup) {
            // Skip if there's only one permission with this normalized name
            if (count($permissionGroup) <= 1) {
                continue;
            }
            
            // Sort permissions to prioritize the one with spaces
            usort($permissionGroup, function($a, $b) {
                // Keep the one with spaces (no hyphens) as the primary one
                return strpos($a->name, '-') !== false ? 1 : -1;
            });
            
            // The first permission is the one we'll keep
            $primaryPermission = $permissionGroup[0];
            
            // Log what we're doing
            echo "Standardizing permission: {$primaryPermission->name}\n";
            
            // Process the duplicate permissions
            for ($i = 1; $i < count($permissionGroup); $i++) {
                $duplicatePermission = $permissionGroup[$i];
                
                echo "  - Removing duplicate: {$duplicatePermission->name}\n";
                
                // Handle model_has_permissions table
                $modelPermissions = DB::table('model_has_permissions')
                    ->where('permission_id', $duplicatePermission->id)
                    ->get();
                
                foreach ($modelPermissions as $modelPerm) {
                    // Check if this role-permission combination already exists for the primary permission
                    $exists = DB::table('model_has_permissions')
                        ->where('permission_id', $primaryPermission->id)
                        ->where('model_id', $modelPerm->model_id)
                        ->where('model_type', $modelPerm->model_type)
                        ->exists();
                    
                    if (!$exists) {
                        // If it doesn't exist, create a new entry with the primary permission
                        DB::table('model_has_permissions')->insert([
                            'permission_id' => $primaryPermission->id,
                            'model_id' => $modelPerm->model_id,
                            'model_type' => $modelPerm->model_type
                        ]);
                    }
                    
                    // Delete the old entry
                    DB::table('model_has_permissions')
                        ->where('permission_id', $duplicatePermission->id)
                        ->where('model_id', $modelPerm->model_id)
                        ->where('model_type', $modelPerm->model_type)
                        ->delete();
                }
                
                // Handle role_has_permissions table
                $rolePermissions = DB::table('role_has_permissions')
                    ->where('permission_id', $duplicatePermission->id)
                    ->get();
                
                foreach ($rolePermissions as $rolePerm) {
                    // Check if this role-permission combination already exists for the primary permission
                    $exists = DB::table('role_has_permissions')
                        ->where('permission_id', $primaryPermission->id)
                        ->where('role_id', $rolePerm->role_id)
                        ->exists();
                    
                    if (!$exists) {
                        // If it doesn't exist, create a new entry with the primary permission
                        DB::table('role_has_permissions')->insert([
                            'permission_id' => $primaryPermission->id,
                            'role_id' => $rolePerm->role_id
                        ]);
                    }
                    
                    // Delete the old entry
                    DB::table('role_has_permissions')
                        ->where('permission_id', $duplicatePermission->id)
                        ->where('role_id', $rolePerm->role_id)
                        ->delete();
                }
                
                // Delete the duplicate permission
                $duplicatePermission->delete();
            }
        }
        
        // Step 2: Remove unused permissions
        // Define the list of permissions we want to keep (all valid permissions)
        $validPermissions = [
            // Basic resource permissions
            'view dashboard',
            'view products', 'create products', 'edit products', 'delete products', 'manage products',
            'view sales', 'create sales', 'edit sales', 'delete sales', 'manage sales', 'process sales',
            'view purchases', 'create purchases', 'edit purchases', 'delete purchases', 'manage purchases',
            'view expenses', 'create expenses', 'edit expenses', 'delete expenses', 'manage expenses',
            'view clients', 'create clients', 'edit clients', 'delete clients', 'manage clients',
            'view suppliers', 'create suppliers', 'edit suppliers', 'delete suppliers', 'manage suppliers',
            'view reports', 'manage reports',
            'view users', 'create users', 'edit users', 'delete users', 'manage users',
            'record payments', 'manage payments',
            'view settings', 'manage settings',
            
            // Special permissions
            'access superadmin panel', 'manage tenants',
            'process purchase returns', 'process sale returns',
            'view audit logs', 'manage backups', 'manage system settings'
        ];
        
        // Get all current permissions
        $currentPermissions = Permission::all();
        
        // Delete permissions that are not in the valid list
        foreach ($currentPermissions as $permission) {
            $normalizedName = str_replace('-', ' ', $permission->name);
            if (!in_array($normalizedName, $validPermissions)) {
                echo "Removing unused permission: {$permission->name}\n";
                
                // Remove all references to this permission
                DB::table('model_has_permissions')
                    ->where('permission_id', $permission->id)
                    ->delete();
                
                DB::table('role_has_permissions')
                    ->where('permission_id', $permission->id)
                    ->delete();
                
                // Delete the permission
                $permission->delete();
            }
        }
    }

    /**
     * Reverse the migrations.
     */
    public function down(): void
    {
        // This migration cannot be reversed as it removes duplicate data
        // We would need the original data to restore it
    }
};
