<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\SoftDeletes;
use Spatie\Multitenancy\Models\Concerns\UsesTenantConnection;

class Sale extends BaseModel
{
    use HasFactory, UsesTenantConnection, SoftDeletes;
    
    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'tenant_id',
        'user_id',
        'client_id',
        'total_amount',
        'discount_amount',
        'final_amount',
        'status',
    ];
    
    /**
     * The attributes that should be cast.
     *
     * @var array<string, string>
     */
    protected $casts = [
        'total_amount' => 'decimal:2',
        'discount_amount' => 'decimal:2',
        'final_amount' => 'decimal:2',
    ];
    
    /**
     * Get the tenant that owns the sale.
     */
    public function tenant()
    {
        return $this->belongsTo(Tenant::class);
    }
    
    /**
     * Get the user that created the sale.
     */
    public function user()
    {
        return $this->belongsTo(User::class);
    }
    
    /**
     * Get the items for the sale.
     */
    public function saleItems()
    {
        return $this->hasMany(SaleItem::class);
    }
    
    /**
     * Calculate the total amount of the sale from its items.
     *
     * @return float
     */
    public function calculateTotalAmount()
    {
        $total = 0;
        
        foreach ($this->saleItems as $item) {
            $total += $item->price_at_sale * $item->quantity;
        }
        
        return $total;
    }
    
    /**
     * Get the raw total amount without any automatic calculation.
     * This ensures we always get the actual stored value.
     *
     * @return float
     */
    public function getRawTotalAmountAttribute()
    {
        return $this->attributes['total_amount'] ?? 0;
    }
    
    /**
     * Get the client that owns the sale.
     */
    public function client()
    {
        return $this->belongsTo(Client::class);
    }
    
    /**
     * Get the returns for the sale.
     */
    public function returns()
    {
        return $this->hasMany(SaleReturn::class);
    }
}
