<?php

namespace App\Notifications;

use App\Models\Payment;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;

class CheckDueNotification extends Notification implements ShouldQueue
{
    use Queueable;

    /**
     * The payment instance.
     *
     * @var \App\Models\Payment
     */
    protected $payment;

    /**
     * Create a new notification instance.
     */
    public function __construct(Payment $payment)
    {
        $this->payment = $payment;
    }

    /**
     * Get the notification's delivery channels.
     *
     * @return array<int, string>
     */
    public function via(object $notifiable): array
    {
        return ['mail', 'database'];
    }

    /**
     * Get the mail representation of the notification.
     */
    public function toMail(object $notifiable): MailMessage
    {
        $payableType = class_basename($this->payment->payable_type);
        $payableName = $this->payment->payable->name ?? 'Unknown';
        $dueDate = $this->payment->check_due_date->format('Y-m-d');
        $amount = number_format($this->payment->amount, 2);
        $checkNumber = $this->payment->check_number ?? 'N/A';
        
        return (new MailMessage)
                    ->subject("Check Due Notification - Due on {$dueDate}")
                    ->greeting("Hello {$notifiable->name},")
                    ->line("This is a reminder that a check is due for processing.")
                    ->line("Check Details:")
                    ->line("- Check Number: {$checkNumber}")
                    ->line("- Amount: {$amount} MAD")
                    ->line("- Due Date: {$dueDate}")
                    ->line("- {$payableType}: {$payableName}")
                    ->action('View Payment Details', url('/payments/' . $this->payment->id))
                    ->line('Please ensure this check is processed on time.');
    }

    /**
     * Get the array representation of the notification.
     *
     * @return array<string, mixed>
     */
    public function toArray(object $notifiable): array
    {
        $payableType = class_basename($this->payment->payable_type);
        $payableName = $this->payment->payable->name ?? 'Unknown';
        
        return [
            'payment_id' => $this->payment->id,
            'check_number' => $this->payment->check_number,
            'amount' => $this->payment->amount,
            'due_date' => $this->payment->check_due_date->format('Y-m-d'),
            'payable_type' => $payableType,
            'payable_name' => $payableName,
            'message' => "Check #{$this->payment->check_number} for {$payableType} '{$payableName}' is due on {$this->payment->check_due_date->format('Y-m-d')}",
        ];
    }
}
