<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Support\Facades\DB;
use Spatie\Multitenancy\Models\Concerns\UsesTenantConnection;
use App\Traits\TenantActivityLogging;

class Product extends BaseModel
{
    use HasFactory, UsesTenantConnection, TenantActivityLogging;
    
    /**
     * Attributes to log in activity
     */
    protected $logAttributes = [
        'name',
        'description',
        'sku',
        'barcode',
        'price',
        'purchase_price',
        'quantity',
        'reorder_point',
        'category_id',
    ];
    
    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'tenant_id',
        'name',
        'description',
        'sku',
        'barcode',
        'price',
        'purchase_price',
        'quantity',
        'reorder_point',
        'category_id',
    ];
    
    /**
     * The attributes that should be cast.
     *
     * @var array<string, string>
     */
    protected $casts = [
        'price' => 'decimal:2',
        'purchase_price' => 'decimal:2',
        'quantity' => 'integer',
        'reorder_point' => 'integer',
    ];
    
    /**
     * Get the tenant that owns the product.
     */
    public function tenant()
    {
        return $this->belongsTo(Tenant::class);
    }
    
    /**
     * Get the category that owns the product.
     */
    public function category()
    {
        return $this->belongsTo(Category::class);
    }
    
    /**
     * Get the sale items for the product.
     */
    public function saleItems()
    {
        return $this->hasMany(SaleItem::class);
    }
    
    /**
     * Get the inventory layers for the product.
     */
    public function inventoryLayers()
    {
        return $this->hasMany(InventoryLayer::class);
    }
    
    /**
     * Get the active inventory layers (with remaining quantity) for the product.
     */
    public function activeInventoryLayers()
    {
        return $this->inventoryLayers()
            ->where('quantity_remaining', '>', 0)
            ->orderBy('created_at', 'asc'); // FIFO order
    }
    
    /**
     * Calculate the average cost of the product based on inventory layers.
     * 
     * @return float
     */
    public function calculateAverageCost()
    {
        $layers = $this->activeInventoryLayers()->get();
        
        if ($layers->isEmpty()) {
            return $this->purchase_price ?? 0;
        }
        
        $totalQuantity = $layers->sum('quantity_remaining');
        $totalValue = $layers->sum(function ($layer) {
            return $layer->quantity_remaining * $layer->unit_cost;
        });
        
        return $totalQuantity > 0 ? $totalValue / $totalQuantity : 0;
    }
    
    /**
     * Get the FIFO cost for a specific quantity of this product.
     * 
     * @param float $quantity
     * @return array Returns [totalCost, costBreakdown]
     */
    public function getFifoCost($quantity)
    {
        $remainingQuantity = $quantity;
        $totalCost = 0;
        $costBreakdown = [];
        
        // Get layers in FIFO order
        $layers = $this->activeInventoryLayers()->get();
        
        foreach ($layers as $layer) {
            if ($remainingQuantity <= 0) {
                break;
            }
            
            $quantityFromLayer = min($remainingQuantity, $layer->quantity_remaining);
            $costFromLayer = $quantityFromLayer * $layer->unit_cost;
            
            $totalCost += $costFromLayer;
            $remainingQuantity -= $quantityFromLayer;
            
            $costBreakdown[] = [
                'layer_id' => $layer->id,
                'quantity' => $quantityFromLayer,
                'unit_cost' => $layer->unit_cost,
                'total_cost' => $costFromLayer
            ];
        }
        
        // If we still have remaining quantity, use the last known purchase price
        if ($remainingQuantity > 0) {
            $lastCost = $this->purchase_price ?? 0;
            $additionalCost = $remainingQuantity * $lastCost;
            $totalCost += $additionalCost;
            
            $costBreakdown[] = [
                'layer_id' => null,
                'quantity' => $remainingQuantity,
                'unit_cost' => $lastCost,
                'total_cost' => $additionalCost
            ];
        }
        
        return [
            'total_cost' => $totalCost,
            'cost_breakdown' => $costBreakdown
        ];
    }
    
    /**
     * Reduce inventory using FIFO method.
     * 
     * @param float $quantity
     * @return array Returns cost breakdown
     */
    public function reduceFifoInventory($quantity)
    {
        $remainingQuantity = $quantity;
        $costBreakdown = [];
        
        DB::beginTransaction();
        
        try {
            // Get layers in FIFO order
            $layers = $this->activeInventoryLayers()->lockForUpdate()->get();
            
            foreach ($layers as $layer) {
                if ($remainingQuantity <= 0) {
                    break;
                }
                
                $quantityFromLayer = min($remainingQuantity, $layer->quantity_remaining);
                $costFromLayer = $quantityFromLayer * $layer->unit_cost;
                
                // Update the layer
                $layer->quantity_remaining -= $quantityFromLayer;
                $layer->save();
                
                $remainingQuantity -= $quantityFromLayer;
                
                $costBreakdown[] = [
                    'layer_id' => $layer->id,
                    'quantity' => $quantityFromLayer,
                    'unit_cost' => $layer->unit_cost,
                    'total_cost' => $costFromLayer
                ];
            }
            
            // If we still have remaining quantity (not enough in layers)
            if ($remainingQuantity > 0) {
                $lastCost = $this->purchase_price ?? 0;
                $additionalCost = $remainingQuantity * $lastCost;
                
                $costBreakdown[] = [
                    'layer_id' => null,
                    'quantity' => $remainingQuantity,
                    'unit_cost' => $lastCost,
                    'total_cost' => $additionalCost
                ];
            }
            
            // Update the product quantity
            $this->quantity -= $quantity;
            $this->save();
            
            DB::commit();
            return $costBreakdown;
        } catch (\Exception $e) {
            DB::rollBack();
            throw $e;
        }
    }
    
    /**
     * Add inventory layer for a purchase.
     * 
     * @param int $purchaseId
     * @param int $purchaseItemId
     * @param float $quantity
     * @param float $unitCost
     * @return InventoryLayer
     */
    public function addInventoryLayer($purchaseId, $purchaseItemId, $quantity, $unitCost)
    {
        return $this->inventoryLayers()->create([
            'tenant_id' => $this->tenant_id,
            'purchase_id' => $purchaseId,
            'purchase_item_id' => $purchaseItemId,
            'quantity_initial' => $quantity,
            'quantity_remaining' => $quantity,
            'unit_cost' => $unitCost,
            'created_at' => now(),
        ]);
    }
}
