<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Spatie\Multitenancy\Models\Concerns\UsesTenantConnection;

class Payment extends Model
{
    use HasFactory, UsesTenantConnection;
    
    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'tenant_id',
        'paymentable_id',
        'paymentable_type',
        'sale_id',
        'user_id',
        'amount',
        'method',
        'payment_date',
        'payment_type',
        'bank_transfer_reference',
        'bank_transfer_receipt_path',
        'check_number',
        'check_due_date',
        'check_status',
        'notes',
    ];
    
    /**
     * The attributes that should be cast.
     *
     * @var array<string, string>
     */
    protected $casts = [
        'payment_date' => 'date',
        'check_due_date' => 'date',
    ];
    
    /**
     * Get the tenant that owns the payment.
     */
    public function tenant()
    {
        return $this->belongsTo(Tenant::class);
    }
    
    /**
     * Get the user who recorded the payment.
     */
    public function user()
    {
        return $this->belongsTo(User::class);
    }
    
    /**
     * Get the paymentable model (client or supplier).
     */
    public function paymentable()
    {
        return $this->morphTo();
    }
    
    /**
     * Get the sale that this payment is for (if it's a sale payment).
     */
    public function sale()
    {
        return $this->belongsTo(Sale::class);
    }
    
    /**
     * Check if this is a sale-specific payment.
     */
    public function isSalePayment()
    {
        return $this->payment_type === 'sale_payment' && !is_null($this->sale_id);
    }
    
    /**
     * Check if this is an account payment.
     */
    public function isAccountPayment()
    {
        return $this->payment_type === 'account_payment' && is_null($this->sale_id);
    }
    
    /**
     * Check if this is a check payment.
     */
    public function isCheckPayment()
    {
        return $this->method === 'check';
    }
    
    /**
     * Check if this payment should be counted in totals.
     * Only count if it's not a check or if it's a paid/pending check.
     */
    public function shouldCountInTotals()
    {
        if (!$this->isCheckPayment()) {
            return true;
        }
        
        return in_array($this->check_status, ['paid', 'pending']);
    }
    
    /**
     * Get due checks (checks that are due today or overdue).
     */
    public static function getDueChecks()
    {
        return static::where('method', 'check')
                    ->where('check_status', 'pending')
                    ->where('check_due_date', '<=', now()->toDateString())
                    ->with(['paymentable', 'sale', 'user'])
                    ->orderBy('check_due_date')
                    ->get();
    }
    
    /**
     * Get all check payments with their status.
     */
    public static function getAllChecks()
    {
        return static::where('method', 'check')
                    ->with(['paymentable', 'sale', 'user'])
                    ->orderBy('check_due_date', 'desc')
                    ->get();
    }
    
    /**
     * Mark check as paid.
     */
    public function markCheckAsPaid()
    {
        if ($this->isCheckPayment()) {
            $this->update(['check_status' => 'paid']);
        }
    }
    
    /**
     * Mark check as unpaid.
     */
    public function markCheckAsUnpaid()
    {
        if ($this->isCheckPayment()) {
            $this->update(['check_status' => 'unpaid']);
        }
    }
    
    /**
     * Mark check as pending.
     */
    public function markCheckAsPending()
    {
        if ($this->isCheckPayment()) {
            $this->update(['check_status' => 'pending']);
        }
    }
}
