<?php

namespace App\Http\Requests;

use App\Models\Sale;
use App\Models\SaleItem;
use Illuminate\Foundation\Http\FormRequest;
use Illuminate\Support\Facades\Auth;
use Illuminate\Validation\Rule;

class StoreSaleReturnRequest extends FormRequest
{
    /**
     * Determine if the user is authorized to make this request.
     */
    public function authorize(): bool
    {
        return $this->user()->can('process-sales-returns');
    }

    /**
     * Get the validation rules that apply to the request.
     *
     * @return array<string, \Illuminate\Contracts\Validation\ValidationRule|array<mixed>|string>
     */
    public function rules(): array
    {
        $saleId = $this->route('sale');
        $sale = Sale::where('id', $saleId)
            ->where('tenant_id', Auth::user()->tenant_id)
            ->first();
            
        if (!$sale) {
            return [];
        }
        
        return [
            'reason' => 'nullable|string',
            'items' => 'required|array|min:1',
            'items.*.sale_item_id' => [
                'required',
                Rule::exists('sale_items', 'id')
                    ->where('sale_id', $sale->id)
                    ->where('tenant_id', Auth::user()->tenant_id),
            ],
            'items.*.quantity' => [
                'required',
                'integer',
                'min:1',
                function ($attribute, $value, $fail) {
                    $index = explode('.', $attribute)[1];
                    $saleItemId = $this->input("items.{$index}.sale_item_id");
                    
                    if (!$saleItemId) {
                        return;
                    }
                    
                    $saleItem = SaleItem::find($saleItemId);
                    if (!$saleItem) {
                        return;
                    }
                    
                    // Calculate already returned quantity for this sale item
                    $alreadyReturnedQty = $saleItem->returnItems()->sum('quantity');
                    $maxReturnableQty = $saleItem->quantity - $alreadyReturnedQty;
                    
                    if ($value > $maxReturnableQty) {
                        $fail("Cannot return more than {$maxReturnableQty} units for this item.");
                    }
                },
            ],
        ];
    }
}
