<?php

namespace App\Http\Requests;

use Illuminate\Foundation\Http\FormRequest;
use Illuminate\Support\Facades\Auth;

class FinalizePOSSaleRequest extends FormRequest
{
    /**
     * Determine if the user is authorized to make this request.
     */
    public function authorize(): bool
    {
        return Auth::check() && Auth::user()->can('process-sales');
    }

    /**
     * Get the validation rules that apply to the request.
     *
     * @return array<string, \Illuminate\Contracts\Validation\ValidationRule|array<mixed>|string>
     */
    public function rules(): array
    {
        return [
            'client_id' => 'required|exists:clients,id',
            'items' => 'required|array|min:1',
            'items.*.product_id' => 'required|exists:products,id',
            'items.*.quantity' => 'required|integer|min:1',
            'items.*.selling_price' => 'required|numeric|min:0',
            'items.*.purchase_price' => 'nullable|numeric|min:0',
            // Make discount fields more flexible
            'items.*.item_discount_type' => 'nullable|string|in:fixed,percentage',
            'items.*.item_discount_value' => 'nullable|numeric|min:0',
            'discount_amount' => 'nullable|numeric|min:0',
            'is_walk_in_cash_payment' => 'nullable|boolean',
        ];
    }
    
    /**
     * Prepare the data for validation.
     *
     * @return void
     */
    protected function prepareForValidation()
    {
        // Set default values for nullable fields
        if ($this->has('items')) {
            $items = collect($this->items)->map(function ($item) {
                // Ensure discount type is always set to a valid value
                if (empty($item['item_discount_type'])) {
                    $item['item_discount_type'] = 'fixed';
                }
                
                // Ensure discount value is always a number
                if (!isset($item['item_discount_value'])) {
                    $item['item_discount_value'] = 0;
                }
                
                return $item;
            })->toArray();
            
            $this->merge([
                'items' => $items,
                'discount_amount' => $this->discount_amount ?? 0,
                'is_walk_in_cash_payment' => $this->is_walk_in_cash_payment ?? false,
            ]);
        }
    }
}
