<?php

namespace App\Http\Middleware;

use Closure;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Symfony\Component\HttpFoundation\Response;

class StrictPermissionCheck
{
    /**
     * Handle an incoming request.
     *
     * @param  \Closure(\Illuminate\Http\Request): (\Symfony\Component\HttpFoundation\Response)  $next
     */
    public function handle(Request $request, Closure $next, string ...$permissions): Response
    {
        $user = Auth::user();
        
        if (!$user) {
            abort(403, 'Unauthorized access');
        }
        
        // Superadmins bypass all permission checks
        if ($user->is_superadmin) {
            return $next($request);
        }
        
        // For regular users, strictly check permissions
        $hasPermission = false;
        
        foreach ($permissions as $permission) {
            // Check both formats of permission names (with and without hyphen)
            $permissionVariants = [
                $permission,
                str_replace(' ', '-', $permission),
                str_replace('-', ' ', $permission)
            ];
            
            foreach ($permissionVariants as $variant) {
                if ($user->hasPermissionTo($variant)) {
                    $hasPermission = true;
                    break 2;
                }
            }
        }
        
        if (!$hasPermission) {
            // Log the unauthorized access attempt
            \Illuminate\Support\Facades\Log::warning('Unauthorized access attempt', [
                'user_id' => $user->id,
                'user_email' => $user->email,
                'route' => $request->route()->getName(),
                'required_permissions' => $permissions,
                'user_permissions' => $user->getAllPermissions()->pluck('name')->toArray()
            ]);
            
            // Redirect to dashboard with error message
            return redirect()->route('dashboard')->with('error', 'You do not have permission to access this page.');
        }
        
        return $next($request);
    }
}
