<?php

namespace App\Http\Middleware;

use Closure;
use Illuminate\Http\Request;
use Symfony\Component\HttpFoundation\Response;

class EnsureTenantAdmin
{
    /**
     * Handle an incoming request.
     *
     * @param  \Closure(\Illuminate\Http\Request): (\Symfony\Component\HttpFoundation\Response)  $next
     */
    public function handle(Request $request, Closure $next): Response
    {
        $user = $request->user();
        
        // Check if user is authenticated and not a superadmin (superadmins should use superadmin panel)
        if (!$user || $user->is_superadmin) {
            abort(403, 'Unauthorized action. Only tenant administrators can access this resource.');
        }
        
        // Check if user has the admin role in the legacy role field
        $hasLegacyAdminRole = ($user->role === 'admin');
        
        // Check if user has an admin role through Spatie role system
        $hasAdminRole = false;
        $userRoles = $user->getRoleNames()->toArray();
        foreach ($userRoles as $roleName) {
            if (strpos($roleName, 'admin-tenant-') === 0) {
                $hasAdminRole = true;
                break;
            }
        }
        
        // Check if user has admin permissions through any method
        $hasUserManagementPermission = $user->hasAnyPermission(['manage-users', 'edit-users', 'create-users']);
        
        // User must have admin role OR explicit user management permissions
        if (!$hasLegacyAdminRole && !$hasAdminRole && !$hasUserManagementPermission) {
            abort(403, 'Unauthorized action. Only tenant administrators can access this resource.');
        }
        
        return $next($request);
    }
}
