<?php

namespace App\Http\Controllers\Web;

use App\Http\Controllers\Controller;
use App\Models\Payment;
use App\Models\Purchase;
use App\Models\PurchaseReturn;
use App\Models\Supplier;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Gate;
use Inertia\Inertia;
use App\Traits\PermissionCheckTrait;

class SupplierController extends Controller
{
    use PermissionCheckTrait;

    /**
     * Create a new controller instance.
     */
    public function __construct()
    {
        $this->middleware('auth');
    }

    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $this->authorizeAnyPermission(['view-suppliers', 'manage-suppliers']);

        $user = Auth::user();
        $tenantId = $user->tenant_id;

        $query = Supplier::where('tenant_id', $tenantId);

        // Simple search functionality
        if ($request->has('search')) {
            $searchTerm = $request->input('search');
            $query->where(function ($q) use ($searchTerm) {
                $q->where('name', 'like', "%{$searchTerm}%")
                  ->orWhere('email', 'like', "%{$searchTerm}%")
                  ->orWhere('phone', 'like', "%{$searchTerm}%");
            });
        }

        $suppliers = $query->orderBy('name')->paginate(15)->withQueryString();

        $canCreate = $user->canAny(['create-suppliers', 'manage-suppliers']);
        $canEdit = $user->canAny(['edit-suppliers', 'manage-suppliers']);
        $canDelete = $user->canAny(['delete-suppliers', 'manage-suppliers']);

        return Inertia::render('Suppliers/Index', [
            'suppliers' => $suppliers,
            'can' => [
                'create_supplier' => $canCreate,
                'edit_supplier' => $canEdit,
                'delete_supplier' => $canDelete,
            ],
            'filters' => $request->only(['search']),
            'debug' => [
                'permissions' => $user->getAllPermissions()->pluck('name')->toArray(),
                'user_id' => $user->id,
                'user_role' => $user->role,
            ]
        ]);
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $this->authorizeAnyPermission(['create-suppliers', 'manage-suppliers']);
        return Inertia::render('Suppliers/Create');
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $this->authorizeAnyPermission(['create-suppliers', 'manage-suppliers']);

        $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'nullable|email|max:255|unique:suppliers,email,NULL,id,tenant_id,' . Auth::user()->tenant_id,
            'phone' => 'nullable|string|max:20',
            'contact_person' => 'nullable|string|max:255',
            'address' => 'nullable|string|max:255',
            'notes' => 'nullable|string',
        ]);

        $supplier = new Supplier([
            'tenant_id' => Auth::user()->tenant_id,
            'name' => $request->name,
            'email' => $request->email,
            'phone' => $request->phone,
            'contact_person' => $request->contact_person,
            'address' => $request->address,
            'notes' => $request->notes,
            'balance' => 0, // New suppliers start with zero balance
        ]);

        $supplier->save();

        return redirect()->route('suppliers.index')
            ->with('message', 'Supplier created successfully.');
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        $this->authorizeAnyPermission(['view-suppliers', 'manage-suppliers']);

        $supplier = Supplier::where('id', $id)
            ->where('tenant_id', Auth::user()->tenant_id)
            ->firstOrFail();

        // Get purchases for this supplier
        $purchases = Purchase::where('supplier_id', $supplier->id)
            ->with('user')
            ->get();

        // Get purchase returns for this supplier
        $purchaseReturns = PurchaseReturn::where('supplier_id', $supplier->id)
            ->with('user')
            ->get();

        // Get payments for this supplier
        $payments = Payment::where('paymentable_type', Supplier::class)
            ->where('paymentable_id', $supplier->id)
            ->with('user')
            ->get();

        // Calculate totals
        $totalPurchases = $purchases->sum('total_amount');
        $totalReturns = $purchaseReturns->sum('total_amount');
        // Only count valid payments (exclude unpaid checks)
        $totalPayments = $payments->filter(function ($payment) {
            return $payment->shouldCountInTotals();
        })->sum('amount');

        // Combine all transactions into a single collection
        $transactions = collect();

        // Add purchases
        foreach ($purchases as $purchase) {
            $transactions->push([
                'id' => 'purchase_' . $purchase->id,
                'date' => $purchase->purchase_date,
                'type' => 'purchase',
                'description' => 'Purchase #' . $purchase->id,
                'amount' => $purchase->total_amount,
                'user' => $purchase->user,
            ]);
        }

        // Add purchase returns
        foreach ($purchaseReturns as $return) {
            $transactions->push([
                'id' => 'return_' . $return->id,
                'date' => $return->return_date,
                'type' => 'return',
                'description' => 'Purchase Return #' . $return->id,
                'amount' => $return->total_amount,
                'user' => $return->user,
            ]);
        }

        // Add payments
        foreach ($payments as $payment) {
            $description = 'Payment (' . ucfirst($payment->method) . ')';
            
            // Add status information for check payments
            if ($payment->isCheckPayment()) {
                $description .= ' - Check #' . $payment->check_number;
                if ($payment->check_status === 'unpaid') {
                    $description .= ' (Unpaid)';
                } elseif ($payment->check_status === 'pending') {
                    $description .= ' (Pending)';
                } elseif ($payment->check_status === 'paid') {
                    $description .= ' (Paid)';
                }
            }
            
            $transactions->push([
                'id' => 'payment_' . $payment->id,
                'date' => $payment->payment_date,
                'type' => 'payment',
                'description' => $description,
                'amount' => $payment->amount,
                'user' => $payment->user,
                'status' => $payment->isCheckPayment() ? $payment->check_status : 'completed',
                'is_valid' => $payment->shouldCountInTotals(),
            ]);
        }

        // Sort transactions by date (newest first)
        $transactions = $transactions->sortByDesc('date')->values()->all();

        return Inertia::render('Suppliers/Show', [
            'supplier' => $supplier,
            'transactions' => $transactions,
            'totalPurchases' => $totalPurchases,
            'totalReturns' => $totalReturns,
            'totalPayments' => $totalPayments,
            'canManageSuppliers' => Gate::allows('manage-suppliers'),
            'canRecordPayments' => Gate::allows('record-payments'),
        ]);
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(string $id)
    {
        $this->authorizeAnyPermission(['edit-suppliers', 'manage-suppliers']);

        $supplier = Supplier::where('id', $id)
            ->where('tenant_id', Auth::user()->tenant_id)
            ->firstOrFail();

        return Inertia::render('Suppliers/Edit', [
            'supplier' => $supplier
        ]);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id)
    {
        $this->authorizeAnyPermission(['edit-suppliers', 'manage-suppliers']);

        $supplier = Supplier::where('id', $id)
            ->where('tenant_id', Auth::user()->tenant_id)
            ->firstOrFail();

        $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'nullable|email|max:255|unique:suppliers,email,' . $supplier->id . ',id,tenant_id,' . Auth::user()->tenant_id,
            'phone' => 'nullable|string|max:20',
            'contact_person' => 'nullable|string|max:255',
            'address' => 'nullable|string|max:255',
            'notes' => 'nullable|string',
        ]);

        // Update supplier fields
        $supplier->name = $request->name;
        $supplier->email = $request->email;
        $supplier->phone = $request->phone;
        $supplier->contact_person = $request->contact_person;
        $supplier->address = $request->address;
        $supplier->notes = $request->notes;

        $supplier->save();

        return redirect()->route('suppliers.index')
            ->with('message', 'Supplier updated successfully.');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        $this->authorizeAnyPermission(['delete-suppliers', 'manage-suppliers']);

        $supplier = Supplier::where('id', $id)
            ->where('tenant_id', Auth::user()->tenant_id)
            ->firstOrFail();

        // Check if supplier has any purchases
        $purchasesCount = $supplier->purchases()->count();
        if ($purchasesCount > 0) {
            return back()->withErrors([
                'message' => 'Cannot delete supplier with associated purchases. There are ' . $purchasesCount . ' purchases linked to this supplier.'
            ]);
        }

        // Check if supplier has any payments
        $paymentsCount = $supplier->payments()->count();
        if ($paymentsCount > 0) {
            return back()->withErrors([
                'message' => 'Cannot delete supplier with associated payments. There are ' . $paymentsCount . ' payments linked to this supplier.'
            ]);
        }

        // Check if supplier has outstanding balance
        if ($supplier->balance != 0) {
            return back()->withErrors([
                'message' => 'Cannot delete supplier with outstanding balance of ' . $supplier->balance . '.'
            ]);
        }

        $supplier->delete();

        return redirect()->route('suppliers.index')
            ->with('message', 'Supplier deleted successfully.');
    }
}
