<?php

namespace App\Http\Controllers\Web;

use App\Http\Controllers\Controller;
use App\Models\Payment;
use App\Models\Supplier;
use Illuminate\Http\Request;
use Inertia\Inertia;

class SupplierCheckController extends Controller
{
    /**
     * Display a listing of all supplier checks.
     */
    public function index(Request $request)
    {
        $query = Payment::where('method', 'check')
                        ->where('paymentable_type', 'App\\Models\\Supplier')
                        ->with(['paymentable', 'user']);
        
        // Filter by supplier if specified
        if ($request->filled('supplier_id')) {
            $query->where('paymentable_id', $request->supplier_id);
        }
        
        // Filter by status if specified
        if ($request->filled('status') && $request->status !== 'all') {
            $query->where('check_status', $request->status);
        }
        
        $checks = $query->orderBy('check_due_date', 'desc')
                       ->paginate(20)
                       ->withQueryString();
        
        // Get all suppliers for the filter dropdown
        $suppliers = Supplier::select('id', 'name')
                            ->where('tenant_id', auth()->user()->tenant_id)
                            ->orderBy('name')
                            ->get();
        
        // Get statistics from all supplier checks (not filtered)
        $allChecksQuery = Payment::where('method', 'check')
                                ->where('paymentable_type', 'App\\Models\\Supplier')
                                ->with(['paymentable', 'user']);
        
        // Apply supplier filter to statistics if specified
        if ($request->filled('supplier_id')) {
            $allChecksQuery->where('paymentable_id', $request->supplier_id);
        }
        
        $allChecks = $allChecksQuery->get();
        
        $statistics = [
            'pending_count' => $allChecks->where('check_status', 'pending')->count(),
            'paid_count' => $allChecks->where('check_status', 'paid')->count(),
            'unpaid_count' => $allChecks->where('check_status', 'unpaid')->count(),
            'overdue_count' => $allChecks->filter(function ($check) {
                return $check->check_due_date < now()->toDateString() && $check->check_status === 'pending';
            })->count(),
            'pending_amount' => $allChecks->where('check_status', 'pending')->sum('amount'),
            'paid_amount' => $allChecks->where('check_status', 'paid')->sum('amount'),
            'unpaid_amount' => $allChecks->where('check_status', 'unpaid')->sum('amount'),
            'overdue_amount' => $allChecks->filter(function ($check) {
                return $check->check_due_date < now()->toDateString() && $check->check_status === 'pending';
            })->sum('amount'),
        ];
        
        return Inertia::render('SupplierChecks/Index', [
            'checks' => $checks->through(function ($check) {
                return [
                    'id' => $check->id,
                    'amount' => $check->amount,
                    'check_number' => $check->check_number,
                    'check_due_date' => $check->check_due_date,
                    'check_status' => $check->check_status,
                    'payment_date' => $check->payment_date,
                    'paymentable_type' => $check->paymentable_type,
                    'paymentable' => $check->paymentable ? [
                        'id' => $check->paymentable->id,
                        'name' => $check->paymentable->name,
                    ] : null,
                    'user' => [
                        'id' => $check->user->id,
                        'name' => $check->user->name,
                    ],
                    'notes' => $check->notes,
                    'is_overdue' => $check->check_due_date < now()->toDateString() && $check->check_status === 'pending',
                ];
            }),
            'suppliers' => $suppliers,
            'statistics' => $statistics,
            'filters' => [
                'supplier_id' => $request->supplier_id,
                'status' => $request->status ?? 'all',
            ],
        ]);
    }
    
    /**
     * Update check status.
     */
    public function updateStatus(Request $request, Payment $payment)
    {
        $request->validate([
            'status' => 'required|in:paid,pending,unpaid',
        ]);
        
        if (!$payment->isCheckPayment()) {
            return back()->withErrors(['error' => 'This payment is not a check payment.']);
        }
        
        // Ensure this is a supplier check
        if ($payment->paymentable_type !== 'App\\Models\\Supplier') {
            return back()->withErrors(['error' => 'This is not a supplier check payment.']);
        }
        
        $oldStatus = $payment->check_status;
        $newStatus = $request->status;
        
        // Update the payment status
        $payment->update(['check_status' => $newStatus]);
        
        // Adjust supplier balance based on status change
        $supplier = $payment->paymentable;
        
        // If changing from paid/pending to unpaid, increase supplier balance (we owe them more)
        if (in_array($oldStatus, ['paid', 'pending']) && $newStatus === 'unpaid') {
            $supplier->balance += $payment->amount;
            $supplier->save();
        }
        // If changing from unpaid to paid/pending, decrease supplier balance (we owe them less)
        elseif ($oldStatus === 'unpaid' && in_array($newStatus, ['paid', 'pending'])) {
            $supplier->balance -= $payment->amount;
            $supplier->save();
        }
        
        return back()->with('success', 'Supplier check status updated successfully.');
    }
    
    /**
     * Get due supplier checks for dashboard alerts.
     */
    public function getDueChecks()
    {
        $dueChecks = Payment::where('method', 'check')
                           ->where('paymentable_type', 'App\\Models\\Supplier')
                           ->where('check_due_date', '<=', now()->toDateString())
                           ->where('check_status', 'pending')
                           ->with(['paymentable'])
                           ->get();
        
        return response()->json([
            'due_checks' => $dueChecks->map(function ($check) {
                return [
                    'id' => $check->id,
                    'amount' => $check->amount,
                    'check_number' => $check->check_number,
                    'check_due_date' => $check->check_due_date,
                    'paymentable_name' => $check->paymentable ? $check->paymentable->name : 'Unknown',
                    'days_overdue' => now()->diffInDays($check->check_due_date, false),
                ];
            }),
        ]);
    }
}