<?php

namespace App\Http\Controllers\Web;

use App\Http\Controllers\Controller;
use App\Models\Sale;
use App\Models\SaleReturn;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Inertia\Inertia;

class SaleReturnController extends Controller
{
    /**
     * Display a listing of the sale returns.
     *
     * @return \Inertia\Response
     */
    public function index()
    {
        // No permission check needed - accessible to all authenticated users

        $saleReturns = SaleReturn::where('tenant_id', Auth::user()->tenant_id)
            ->with(['client:id,name', 'user:id,name'])
            ->orderBy('created_at', 'desc')
            ->paginate(15);

        return Inertia::render('Sales/Returns/Index', [
            'saleReturns' => $saleReturns,
            'can' => [
                'manage_sales_returns' => Auth::user()->can('manage-sales-returns'),
                'process_sales_returns' => Auth::user()->can('process-sales-returns'),
            ]
        ]);
    }

    /**
     * Display a page for selecting a sale to process a return for.
     *
     * @return \Inertia\Response
     */
    public function selectSale()
    {
        // Check if the user has permission to process sales returns
        if (!Auth::user()->can('process-sales-returns')) {
            abort(403, 'You do not have permission to process sales returns.');
        }
        
        // Get completed sales that can be returned (exclude soft deleted)
        $sales = Sale::withoutTrashed()
            ->where('tenant_id', Auth::user()->tenant_id)
            ->where('status', 'completed')
            ->with(['client:id,name', 'user:id,name'])
            ->orderBy('created_at', 'desc')
            ->paginate(15);
            
        return Inertia::render('Sales/Returns/SelectSale', [
            'sales' => $sales,
            'can' => [
                'process_sales_returns' => Auth::user()->can('process-sales-returns'),
            ]
        ]);
    }

    /**
     * Display the specified sale return.
     *
     * @param  \App\Models\SaleReturn  $saleReturn
     * @return \Inertia\Response
     */
    public function show(SaleReturn $saleReturn)
    {
        // No permission check needed - accessible to all authenticated users

        // Check if the sale return belongs to the current tenant
        if ($saleReturn->tenant_id !== Auth::user()->tenant_id) {
            abort(404);
        }

        $saleReturn->load([
            'sale',
            'client:id,name,email,phone',
            'user:id,name',
            'items.product:id,name,sku',
        ]);

        return Inertia::render('Sales/Returns/Show', [
            'saleReturn' => $saleReturn,
            'can' => [
                'manage_sales_returns' => Auth::user()->can('manage-sales-returns'),
            ]
        ]);
    }
}
