<?php

namespace App\Http\Controllers\Web;

use App\Http\Controllers\Controller;
use App\Models\Client;
use App\Models\Product;
use App\Models\Sale;
use App\Models\SaleItem;
use App\Models\Payment;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Validation\ValidationException;

class SaleFinalizationController extends Controller
{
    /**
     * Finalize a sale
     * 
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function __invoke(Request $request)
    {
        // Validate the request
        $validated = $request->validate([
            'client_id' => 'required|exists:clients,id',
            'items' => 'required|array|min:1',
            'items.*.product_id' => 'required|exists:products,id',
            'items.*.quantity' => 'required|numeric|min:0.01',
            'items.*.selling_price' => 'required|numeric|min:0',
            'discount_amount' => 'nullable|numeric|min:0',
            'payment_type' => 'required|in:full,partial,none',
            'payment' => 'nullable|array',
            'payment.amount' => 'required_if:payment_type,full,partial|numeric|min:0',
            'payment.method' => 'required_if:payment_type,full,partial|in:cash,card,bank_transfer,check',
            'payment.payment_date' => 'required_if:payment_type,full,partial|date',
            'payment.bank_transfer_reference' => 'required_if:payment.method,bank_transfer',
            'payment.bank_transfer_receipt_path' => 'nullable|string',
            'payment.check_number' => 'required_if:payment.method,check',
            'payment.check_due_date' => 'required_if:payment.method,check|date|nullable',
        ]);

        // Start a database transaction
        return DB::transaction(function () use ($validated, $request) {
            $client = Client::findOrFail($validated['client_id']);
            
            // Check if client belongs to the same tenant
            if ($client->tenant_id !== Auth::user()->tenant_id) {
                throw ValidationException::withMessages([
                    'client_id' => ['Client not found.']
                ]);
            }
            
            // Calculate totals first
            $subtotalAmount = 0;
            foreach ($validated['items'] as $item) {
                $itemTotal = (float)$item['selling_price'] * (float)$item['quantity'];
                $subtotalAmount += $itemTotal;
            }
            
            $discountAmount = $validated['discount_amount'] ?? 0;
            $finalAmount = $subtotalAmount - $discountAmount;
            
            // Determine sale status based on payment
            $saleStatus = 'not_paid';
            if ($validated['payment_type'] === 'full') {
                $saleStatus = 'paid';
            } elseif ($validated['payment_type'] === 'partial') {
                $saleStatus = 'partial_paid';
            }
            
            // Create the sale
            $sale = new Sale([
                'tenant_id' => Auth::user()->tenant_id,
                'user_id' => Auth::id(),
                'client_id' => $client->id,
                'total_amount' => $subtotalAmount,
                'discount_amount' => $discountAmount,
                'final_amount' => $finalAmount,
                'status' => $saleStatus,
            ]);
            
            $sale->save();
            
            // Process each item
            foreach ($validated['items'] as $item) {
                $product = Product::where('id', $item['product_id'])
                    ->where('tenant_id', Auth::user()->tenant_id)
                    ->lockForUpdate()
                    ->firstOrFail();
                
                // Check if there's enough stock
                if ($product->quantity < $item['quantity']) {
                    throw ValidationException::withMessages([
                        'items' => ["Not enough stock for product: {$product->name}"]
                    ]);
                }
                
                // Create sale item
                $saleItem = new SaleItem([
                    'tenant_id' => Auth::user()->tenant_id,
                    'sale_id' => $sale->id,
                    'product_id' => $product->id,
                    'quantity' => $item['quantity'],
                    'price_at_sale' => $item['selling_price'],
                ]);
                
                $saleItem->save();
                
                // Update product stock
                $product->quantity -= $item['quantity'];
                $product->save();
            }
            
            // Create payment record if payment was made
            if ($validated['payment_type'] !== 'none' && isset($validated['payment'])) {
                $payment = new Payment([
                    'tenant_id' => Auth::user()->tenant_id,
                    'paymentable_type' => Client::class,
                    'paymentable_id' => $client->id,
                    'sale_id' => $sale->id,
                    'user_id' => Auth::id(),
                    'amount' => $validated['payment']['amount'],
                    'method' => $validated['payment']['method'],
                    'payment_date' => $validated['payment']['payment_date'],
                    'payment_type' => 'sale_payment',
                    'notes' => 'Payment for Sale #' . $sale->id,
                ]);
                
                // Add method-specific fields
                if ($validated['payment']['method'] === 'bank_transfer') {
                    $payment->bank_transfer_reference = $validated['payment']['bank_transfer_reference'] ?? null;
                    $payment->bank_transfer_receipt_path = $validated['payment']['bank_transfer_receipt_path'] ?? null;
                } else if ($validated['payment']['method'] === 'check') {
                    $payment->check_number = $validated['payment']['check_number'] ?? null;
                    $payment->check_due_date = $validated['payment']['check_due_date'] ?? null;
                    $payment->check_status = 'pending'; // Default status for new checks
                }
                
                $payment->save();
                $paidAmount = $validated['payment']['amount'];
                
                // Update sale payment status after payment creation
                $sale->updatePaymentStatus();
            }
            
            // Update client balance with unpaid amount
            $paidAmount = ($validated['payment_type'] !== 'none' && isset($validated['payment'])) 
                ? $validated['payment']['amount'] 
                : 0;
            $unpaidAmount = $finalAmount - $paidAmount;
            
            if ($unpaidAmount > 0) {
                $client->balance += $unpaidAmount;
                $client->save();
            }
            
            return response()->json([
                'message' => 'Sale completed successfully',
                'sale' => $sale,
            ]);
        });
    }
}
