<?php

namespace App\Http\Controllers\Web;

use App\Http\Controllers\Controller;
use App\Models\Client;
use App\Models\Product;
use App\Models\Sale;
use App\Models\SaleItem;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Inertia\Inertia;
use App\Traits\PermissionCheckTrait;


class SaleController extends Controller
{
    use PermissionCheckTrait;

    /**
     * Display a listing of the sales.
     *
     * @return \Inertia\Response
     */
    public function index()
    {
        $this->authorizeAnyPermission(['view-sales', 'manage-sales']);

        // Get sales with necessary relationships for profit calculation
        $sales = Sale::withoutTrashed() // Exclude soft-deleted sales
            ->where('tenant_id', Auth::user()->tenant_id)
            ->with(['client:id,name', 'user:id,name', 'saleItems.product'])
            ->orderBy('created_at', 'desc')
            ->paginate(15);
            
        // Ensure profit calculations are included for each sale
        foreach ($sales as $sale) {
            // If profit data is missing, calculate it
            if ($sale->total_cost == 0 || $sale->gross_profit == 0) {
                // Calculate total cost from sale items
                $totalCost = 0;
                
                foreach ($sale->saleItems as $item) {
                    if ($item->total_cost > 0) {
                        $totalCost += $item->total_cost;
                    } else if ($item->cost_at_sale > 0) {
                        $totalCost += $item->cost_at_sale * $item->quantity;
                    } else if ($item->product && $item->product->purchase_price > 0) {
                        $totalCost += $item->product->purchase_price * $item->quantity;
                    }
                }
                
                // Update the sale with calculated profits
                if ($totalCost > 0) {
                    $sale->total_cost = $totalCost;
                    $sale->gross_profit = $sale->final_amount - $totalCost;
                    $sale->profit_margin_percentage = $sale->final_amount > 0 ? 
                        ($sale->gross_profit / $sale->final_amount) * 100 : 0;
                    $sale->save();
                }
            }
        }

        $user = Auth::user();
        $canCreate = $user->canAny(['create-sales', 'manage-sales']);
        $canEdit = $user->canAny(['edit-sales', 'manage-sales']);
        $canDelete = $user->canAny(['delete-sales', 'manage-sales']);
        $canRecordPayment = $user->canAny(['record-payments', 'manage-payments']);

        return Inertia::render('Sales/Index', [
            'sales' => $sales,
            'can' => [
                'create_sale' => $canCreate,
                'edit_sale' => $canEdit,
                'delete_sale' => $canDelete,
                'record_payment' => $canRecordPayment,
            ],
            'debug' => [
                'permissions' => $user->getAllPermissions()->pluck('name')->toArray(),
                'user_id' => $user->id,
                'user_role' => $user->role,
            ]
        ]);
    }

    /**
     * Show the form for creating a new sale.
     *
     * @return \Inertia\Response
     */
    public function create()
    {
        $this->authorizeAnyPermission(['create-sales', 'process-sales']);

        $clients = Client::where('tenant_id', Auth::user()->tenant_id)
            ->select('id', 'name')
            ->orderBy('name')
            ->get();

        return Inertia::render('Sales/Create', [
            'clients' => $clients,
            'can' => [
                'process_sales' => Auth::user()->hasPermissionTo('process-sales'),
            ]
        ]);
    }

    /**
     * Store a newly created sale in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function store(Request $request)
    {
        $this->authorizeAnyPermission(['create-sales', 'process-sales']);

        $validated = $request->validate([
            'client_id' => 'required|exists:clients,id',
            'items' => 'required|array|min:1',
            'items.*.product_id' => 'required|exists:products,id',
            'items.*.quantity' => 'required|integer|min:1',
            'items.*.price' => 'required|numeric|min:0',
            'notes' => 'nullable|string',
        ]);

        // Start a database transaction
        \DB::beginTransaction();

        try {
            // Calculate total amount
            $totalAmount = 0;
            foreach ($validated['items'] as $item) {
                $totalAmount += $item['quantity'] * $item['price'];
            }
            
            // Handle discount if provided
            $discountAmount = $validated['discount_amount'] ?? 0;
            $finalAmount = $totalAmount - $discountAmount;

            // Create the sale
            $sale = new Sale([
                'tenant_id' => Auth::user()->tenant_id,
                'client_id' => $validated['client_id'],
                'user_id' => Auth::user()->id,
                'total_amount' => $totalAmount,
                'discount_amount' => $discountAmount,
                'final_amount' => $finalAmount,
                'status' => 'completed',
                'notes' => $validated['notes'] ?? null,
            ]);
            $sale->save();
            
            // Track total cost for profitability calculations
            $totalCost = 0;

            // Create the sale items and update inventory
            foreach ($validated['items'] as $item) {
                $product = Product::findOrFail($item['product_id']);
                
                // Check if there's enough stock
                if ($product->quantity < $item['quantity']) {
                    throw new \Exception("Not enough stock for product: {$product->name}");
                }
                
                // Get FIFO cost breakdown and reduce inventory
                $costBreakdown = $product->reduceFifoInventory($item['quantity']);
                
                // Calculate average cost from the FIFO breakdown
                $totalItemCost = 0;
                foreach ($costBreakdown as $layer) {
                    $totalItemCost += $layer['total_cost'];
                }
                
                // If FIFO data doesn't provide a cost (e.g., totalItemCost is 0), 
                // use the product's purchase_price as a fallback
                if ($totalItemCost == 0 && $product->purchase_price > 0) {
                    $totalItemCost = $product->purchase_price * $item['quantity'];
                }
                
                $avgCostPerUnit = $item['quantity'] > 0 ? $totalItemCost / $item['quantity'] : 0;
                
                // Create sale item with cost tracking
                $saleItem = $sale->saleItems()->create([
                    'tenant_id' => Auth::user()->tenant_id,
                    'product_id' => $item['product_id'],
                    'quantity' => $item['quantity'],
                    'price_at_sale' => $item['price'],
                    'cost_at_sale' => $avgCostPerUnit,
                    'unit_profit' => $item['price'] - $avgCostPerUnit,
                    'total_cost' => $totalItemCost,
                    'total_profit' => ($item['price'] - $avgCostPerUnit) * $item['quantity'],
                ]);
                
                // Add to total cost
                $totalCost += $totalItemCost;
            }

            // Update sale with profitability metrics
            $sale->total_cost = $totalCost;
            $sale->gross_profit = $finalAmount - $totalCost;
            $sale->profit_margin_percentage = ($finalAmount > 0) ? (($sale->gross_profit / $finalAmount) * 100) : 0;
            $sale->save();
            
            // Update client balance with the final amount (after discount)
            $client = Client::findOrFail($validated['client_id']);
            $client->balance += $finalAmount;
            $client->save();

            \DB::commit();

            return redirect()->route('sales.show', $sale->id)->with('success', 'Sale created successfully.');
        } catch (\Exception $e) {
            \DB::rollBack();
            return redirect()->back()->with('error', 'Error creating sale: ' . $e->getMessage());
        }
    }

    /**
     * Display the specified sale.
     *
     * @param  \App\Models\Sale  $sale
     * @return \Inertia\Response
     */
    public function show(Sale $sale)
    {
        $this->authorizeAnyPermission(['view-sales', 'manage-sales']);

        // Check if the sale belongs to the current tenant
        if ($sale->tenant_id !== Auth::user()->tenant_id) {
            abort(404);
        }

        // Load all relationships with all fields to ensure we have complete data
        $sale->load([
            'client',
            'user',
            'saleItems.product',
        ]);
        
        // Make sure profitability data is loaded from database
        $sale->refresh();
        
        // Ensure product data is available for each sale item
        foreach ($sale->saleItems as $item) {
            if (!$item->product) {
                // If product relationship isn't loading, try to load it manually
                $product = \App\Models\Product::find($item->product_id);
                if ($product) {
                    $item->product = $product;
                }
            }
        }
        
        // Always recalculate total amount to ensure it's correct
        $calculated_total = $sale->saleItems->sum(function($item) {
            // Check if we should use unit_price or price_at_sale
            $price = $item->unit_price ?? $item->price_at_sale;
            // If we have a total_price field, use that directly
            if (isset($item->total_price) && $item->total_price > 0) {
                return $item->total_price;
            }
            return $item->quantity * $price;
        });
        
        // Update the total amount and ensure profitability data is set
        $sale->total_amount = $calculated_total;
        
        // If profit data is missing, recalculate it
        if ($sale->total_cost == 0 || $sale->gross_profit == 0) {
            // Calculate total cost from sale items
            $totalCost = $sale->saleItems->sum('total_cost');
            
            // If total_cost is still zero, try calculating from the product's purchase_price
            if ($totalCost == 0) {
                $totalCost = $sale->saleItems->sum(function($item) {
                    $product = $item->product;
                    if ($product && $product->purchase_price > 0) {
                        return $product->purchase_price * $item->quantity;
                    }
                    return 0;
                });
            }
            
            // Update the sale with the calculated costs and profits
            $sale->total_cost = $totalCost;
            $sale->gross_profit = $sale->final_amount - $totalCost;
            $sale->profit_margin_percentage = $sale->final_amount > 0 ? 
                ($sale->gross_profit / $sale->final_amount) * 100 : 0;
                
            // Save the updated calculations
            $sale->save();
        }

        // Prepare the sale data with explicit items array
        $saleData = $sale->toArray();
        
        // Make sure we have the sale items explicitly included
        $saleData['saleItems'] = $sale->saleItems->map(function($item) {
            $itemData = $item->toArray();
            // Include product data if available
            if ($item->product) {
                $itemData['product'] = $item->product->toArray();
            } else {
                $itemData['product'] = ['id' => $item->product_id, 'name' => 'Product #' . $item->product_id];
            }
            return $itemData;
        })->values()->all();
        
        // Add debug info in non-production environments
        $debug = null;
        if (!app()->environment('production')) {
            $debug = [
                'user_id' => Auth::id(),
                'user_email' => Auth::user()->email,
                'all_permissions' => Auth::user()->getAllPermissions()->pluck('name'),
                'roles' => Auth::user()->getRoleNames(),
            ];
        }

        return Inertia::render('Sales/Show', [
            'sale' => $saleData,
            'can' => [
                'manage_sales' => auth('web')->user() ? auth('web')->user()->canAny(['manage-sales']) : false,
                'process_sales_returns' => auth('web')->user() ? auth('web')->user()->canAny(['process-sales-returns']) : false,
            ],
            
        ]);
    }
    
    /**
     * Show the form for editing the specified sale.
     *
     * @param  \App\Models\Sale  $sale
     * @return \Inertia\Response
     */
    public function edit(Sale $sale)
    {
        $this->authorizeAnyPermission(['edit-sales', 'manage-sales']);
        
        // Check if the sale belongs to the current tenant
        if ($sale->tenant_id !== Auth::user()->tenant_id) {
            abort(404);
        }
        
        // Don't allow editing of cancelled sales
        if ($sale->status === 'cancelled') {
            return redirect()->route('sales.show', $sale->id)
                ->with('error', 'Cancelled sales cannot be edited.');
        }
        
        // Load only the necessary relationships without products
        $sale->load(['client', 'saleItems']);
        
        // Get all clients for the dropdown
        $clients = Client::where('tenant_id', Auth::user()->tenant_id)
            ->select('id', 'name')
            ->orderBy('name')
            ->get();
        
        // Get all product IDs from sale items
        $productIds = $sale->saleItems->pluck('product_id')->unique()->toArray();
        
        // Fetch fresh product data directly from database
        $products = Product::whereIn('id', $productIds)->get();
        
        // Create a lookup array for products
        $productLookup = [];
        foreach ($products as $product) {
            $productLookup[$product->id] = [
                'id' => $product->id,
                'name' => $product->name,
                'stock_quantity' => $product->quantity, // Using 'quantity' instead of 'stock_quantity'
                'selling_price' => $product->price     // Using 'price' instead of 'selling_price'
            ];
        }
        
        // Prepare sale items with fresh product data
        $saleItems = [];
        foreach ($sale->saleItems as $item) {
            $productData = $productLookup[$item->product_id] ?? [
                'id' => $item->product_id,
                'name' => 'Product #' . $item->product_id,
                'stock_quantity' => 0,
                'selling_price' => 0
            ];
            
            $saleItems[] = [
                'id' => $item->id,
                'product_id' => $item->product_id,
                'quantity' => $item->quantity,
                'price_at_sale' => $item->price_at_sale,
                'product' => $productData
            ];
        }
        
        // Prepare the sale data
        $saleData = [
            'id' => $sale->id,
            'client_id' => $sale->client_id,
            'status' => $sale->status,
            'total_amount' => $sale->total_amount,
            'created_at' => $sale->created_at,
            'updated_at' => $sale->updated_at,
            'saleItems' => $saleItems
        ];
        
        return Inertia::render('Sales/Edit', [
            'sale' => $saleData,
            'clients' => $clients,
            'can' => [
                'manage_sales' => Auth::user()->hasPermissionTo('manage-sales')
            ]
        ]);
    }
    
    /**
     * Update the specified sale in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\Models\Sale  $sale
     * @return \Illuminate\Http\RedirectResponse
     */
    public function update(Request $request, Sale $sale)
    {
        $this->authorizeAnyPermission(['edit-sales', 'manage-sales']);
        
        // Check if the sale belongs to the current tenant
        if ($sale->tenant_id !== Auth::user()->tenant_id) {
            abort(404);
        }
        
        // Don't allow editing of cancelled sales
        if ($sale->status === 'cancelled') {
            return redirect()->route('sales.show', $sale->id)
                ->with('error', 'Cancelled sales cannot be edited.');
        }
        
        $validated = $request->validate([
            'client_id' => 'required|exists:clients,id',
            'items' => 'required|array',
            'items.*.id' => 'nullable|integer|exists:sale_items,id',
            'items.*.product_id' => 'required|integer|exists:products,id',
            'items.*.quantity' => 'required|integer|min:1',
            'items.*.selling_price' => 'required|numeric|min:0',
            'items.*.is_original' => 'required|boolean',
        ]);
        
        // Use a database transaction to ensure atomicity
        return \DB::transaction(function () use ($sale, $validated, $request) {
            // Load the sale with its items
            $sale->load('saleItems');
            
            // Get all products involved in the sale
            $productIds = collect($validated['items'])->pluck('product_id')->unique()->toArray();
            $products = Product::whereIn('id', $productIds)->get()->keyBy('id');
            
            // Calculate old total amount for balance adjustment
            $oldTotalAmount = $sale->total_amount;
            
            // Track items to keep, update, and add
            $existingItemIds = $sale->saleItems->pluck('id')->toArray();
            $updatedItemIds = [];
            $newTotalAmount = 0;
            
            // Process each item in the request
            foreach ($validated['items'] as $itemData) {
                $isOriginal = $itemData['is_original'] ?? false;
                $product = $products[$itemData['product_id']] ?? null;
                
                if (!$product) {
                    continue; // Skip if product not found
                }
                
                // Calculate line total
                $lineTotal = $itemData['quantity'] * $itemData['selling_price'];
                $newTotalAmount += $lineTotal;
                
                if ($isOriginal && isset($itemData['id'])) {
                    // This is an existing item - update it
                    $saleItem = SaleItem::where('id', $itemData['id'])
                        ->where('sale_id', $sale->id)
                        ->first();
                    
                    if ($saleItem) {
                        // Calculate quantity difference for inventory adjustment
                        $quantityDiff = $itemData['quantity'] - $saleItem->quantity;
                        
                        // Update the sale item
                        $saleItem->quantity = $itemData['quantity'];
                        $saleItem->price_at_sale = $itemData['selling_price'];
                        $saleItem->save();
                        
                        // Adjust inventory if quantity changed
                        if ($quantityDiff != 0) {
                            // If quantity increased, reduce inventory
                            // If quantity decreased, increase inventory
                            $product->quantity -= $quantityDiff;
                            $product->save();
                        }
                        
                        $updatedItemIds[] = $saleItem->id;
                    }
                } else {
                    // This is a new item - add it
                    $saleItem = new SaleItem([
                        'tenant_id' => Auth::user()->tenant_id,
                        'sale_id' => $sale->id,
                        'product_id' => $itemData['product_id'],
                        'quantity' => $itemData['quantity'],
                        'price_at_sale' => $itemData['selling_price'],
                    ]);
                    $saleItem->save();
                    
                    // Reduce inventory for new items
                    $product->quantity -= $itemData['quantity'];
                    $product->save();
                    
                    $updatedItemIds[] = $saleItem->id;
                }
            }
            
            // Find items to remove (items that exist but weren't in the update request)
            $itemsToRemove = $sale->saleItems->whereNotIn('id', $updatedItemIds);
            
            // Process removed items
            foreach ($itemsToRemove as $itemToRemove) {
                // Restore inventory
                $product = Product::find($itemToRemove->product_id);
                if ($product) {
                    $product->quantity += $itemToRemove->quantity;
                    $product->save();
                }
                
                // Delete the item
                $itemToRemove->delete();
            }
            
            // Update the sale with new total amount
            $sale->total_amount = $newTotalAmount;
            
            // Check if client has changed
            $clientChanged = $sale->client_id != $validated['client_id'];
            
            if ($clientChanged) {
                // Load the old and new clients
                $oldClient = Client::findOrFail($sale->client_id);
                $newClient = Client::findOrFail($validated['client_id']);
                
                // Adjust the old client's balance (they owe less)
                $oldClient->balance -= $oldTotalAmount;
                $oldClient->save();
                
                // Adjust the new client's balance (they owe more)
                $newClient->balance += $newTotalAmount;
                $newClient->save();
            } else {
                // Same client but total amount changed
                if ($oldTotalAmount != $newTotalAmount) {
                    $client = Client::findOrFail($sale->client_id);
                    // Adjust the client's balance by the difference
                    $client->balance = $client->balance - $oldTotalAmount + $newTotalAmount;
                    $client->save();
                }
            }
            
            // Update the sale
            $sale->client_id = $validated['client_id'];
            $sale->save();
            
            return redirect()->route('sales.show', $sale->id)
                ->with('success', 'Sale updated successfully.');
        });
    }
    
    /**
     * Remove the specified sale from storage.
     *
     * @param  \App\Models\Sale  $sale
     * @return \Illuminate\Http\RedirectResponse
     */
    public function destroy(Sale $sale)
    {
        $this->authorizeAnyPermission(['delete-sales', 'manage-sales']);
        
        // Check if the sale belongs to the current tenant
        if ($sale->tenant_id !== Auth::user()->tenant_id) {
            abort(404);
        }
        
        // Use a database transaction to ensure atomicity
        return \DB::transaction(function () use ($sale) {
            // Load the sale items if they haven't been loaded yet
            if (!$sale->relationLoaded('saleItems')) {
                $sale->load('saleItems.product');
            }
            
            // Return inventory for each sale item
            foreach ($sale->saleItems as $item) {
                $product = $item->product;
                if ($product) {
                    // Increase the product quantity (return to inventory)
                    $product->quantity += $item->quantity;
                    $product->save();
                }
            }
            
            // Adjust client balance (client owes less)
            $client = $sale->client;
            if ($client) {
                $client->balance -= $sale->total_amount;
                $client->save();
            }
            
            // Soft delete the sale (this will set deleted_at timestamp)
            $sale->delete();
            
            return redirect()->route('sales.index')->with('success', 'Sale deleted successfully. Inventory has been restored.');
        });
    }
}
