<?php

namespace App\Http\Controllers\Web;

use App\Http\Controllers\Controller;
use App\Models\Purchase;
use App\Models\PurchaseReturn;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Inertia\Inertia;

class PurchaseReturnController extends Controller
{
    /**
     * Display a listing of the purchase returns.
     *
     * @return \Inertia\Response
     */
    public function index()
    {
        // No permission check needed - accessible to all authenticated users

        $purchaseReturns = PurchaseReturn::where('tenant_id', Auth::user()->tenant_id)
            ->with(['supplier:id,name', 'user:id,name'])
            ->orderBy('created_at', 'desc')
            ->paginate(15);

        return Inertia::render('Purchases/Returns/Index', [
            'purchaseReturns' => $purchaseReturns,
            'can' => [
                'manage_purchase_returns' => Auth::user()->hasAnyPermission(['manage purchase returns', 'manage-purchase-returns']),
                'process_purchase_returns' => Auth::user()->hasAnyPermission(['process-purchase-returns', 'process-purchase-returns']),
            ]
        ]);
    }

    /**
     * Display a page for selecting a purchase to process a return for.
     *
     * @return \Inertia\Response
     */
    public function selectPurchase()
    {
        // Check if the user has permission to process purchase returns
        if (!Auth::user()->hasAnyPermission(['process-purchase-returns', 'process-purchase-returns'])) {
            abort(403, 'You do not have permission to process purchase returns.');
        }
        
        // Get completed purchases that can be returned
        $purchases = Purchase::where('tenant_id', Auth::user()->tenant_id)
            ->where('status', 'completed')
            ->with(['supplier:id,name', 'user:id,name'])
            ->orderBy('created_at', 'desc')
            ->paginate(15);
            
        return Inertia::render('Purchases/Returns/SelectPurchase', [
            'purchases' => $purchases,
            'can' => [
                'process_purchase_returns' => Auth::user()->can('process-purchase-returns'),
            ]
        ]);
    }

    /**
     * Display the specified purchase return.
     *
     * @param  \App\Models\PurchaseReturn  $purchaseReturn
     * @return \Inertia\Response
     */
    public function show(PurchaseReturn $purchaseReturn)
    {
        // No permission check needed - accessible to all authenticated users

        // Check if the purchase return belongs to the current tenant
        if ($purchaseReturn->tenant_id !== Auth::user()->tenant_id) {
            abort(404);
        }

        $purchaseReturn->load([
            'purchase',
            'supplier:id,name,email,phone',
            'user:id,name',
            'items.product:id,name,sku',
        ]);

        return Inertia::render('Purchases/Returns/Show', [
            'purchaseReturn' => $purchaseReturn,
            'can' => [
                'manage_purchase_returns' => Auth::user()->can('manage-purchase-returns'),
            ]
        ]);
    }
}
