<?php

namespace App\Http\Controllers\Web;

use App\Http\Controllers\Controller;
use App\Http\Requests\StorePurchaseRequest;
use App\Models\Product;
use App\Models\Purchase;
use App\Models\PurchaseItem;
use App\Models\Supplier;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Inertia\Inertia;
use App\Traits\PermissionCheckTrait;

class PurchaseController extends Controller
{
    use PermissionCheckTrait;

    /**
     * Create a new controller instance.
     */
    public function __construct()
    {
        // Removed middleware, using PermissionCheckTrait instead
    }
    
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        // Check for any of these permissions with various formats
        $this->authorizeAnyPermission([
            'view-purchases', 
            'view-purchases',
            'manage-purchases', 
            'manage-purchases',
            'view purchase', 
            'view-purchase',
            'manage purchase', 
            'manage-purchase'
        ]);

        $user = Auth::user();
        $tenantId = $user->tenant_id;

        $query = Purchase::with(['supplier:id,name', 'user:id,name'])
                         ->where('tenant_id', $tenantId);

        // Search functionality (example: search by purchase ID, supplier name)
        if ($request->has('search')) {
            $searchTerm = $request->input('search');
            $query->where(function ($q) use ($searchTerm) {
                $q->where('id', 'like', "%{$searchTerm}%")
                  ->orWhereHas('supplier', function ($supplierQuery) use ($searchTerm) {
                      $supplierQuery->where('name', 'like', "%{$searchTerm}%");
                  });
            });
        }

        // Add date filtering if needed
        if ($request->filled('start_date') && $request->filled('end_date')) {
            $query->whereBetween('created_at', [$request->start_date . ' 00:00:00', $request->end_date . ' 23:59:59']);
        }

        $purchases = $query->orderBy('created_at', 'desc')->paginate(15)->withQueryString();

        // Determine UI capabilities - handle admin users specially
        if ($user->role === 'admin') {
            // Admin users have all permissions except superadmin ones
            $canCreate = true;
            $canEdit = true;
            $canDelete = true;
            $canRecordPayment = true;
        } else {
            // For non-admin users, check permissions explicitly
            $canCreate = $user->hasAnyPermission([
                'create-purchases', 'create-purchases', 'create purchase', 'create-purchase',
                'manage-purchases', 'manage-purchases', 'manage purchase', 'manage-purchase'
            ]);
            $canEdit = $user->hasAnyPermission([
                'edit-purchases', 'edit-purchases', 'edit purchase', 'edit-purchase',
                'manage-purchases', 'manage-purchases', 'manage purchase', 'manage-purchase'
            ]);
            $canDelete = $user->hasAnyPermission([
                'delete-purchases', 'delete-purchases', 'delete purchase', 'delete-purchase',
                'manage-purchases', 'manage-purchases', 'manage purchase', 'manage-purchase'
            ]);
            $canRecordPayment = $user->hasAnyPermission([
                'record-payments', 'record-payments', 'record payment', 'record-payment',
                'manage-payments', 'manage-payments', 'manage payment', 'manage-payment'
            ]);
        }

        return Inertia::render('Purchases/Index', [
            'purchases' => $purchases,
            'can' => [
                'create_purchase' => $canCreate,
                'edit_purchase' => $canEdit,
                'delete_purchase' => $canDelete,
                'record_payment' => $canRecordPayment, 
            ],
            'filters' => $request->only(['search', 'start_date', 'end_date']),
        ]);
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        // Check for any of these permissions with various formats
        $this->authorizeAnyPermission([
            'create-purchases', 'create-purchases', 'create purchase', 'create-purchase',
            'manage-purchases', 'manage-purchases', 'manage purchase', 'manage-purchase'
        ]);

        $tenantId = Auth::user()->tenant_id;
        $products = Product::where('tenant_id', $tenantId)->select('id', 'name', 'sku', 'purchase_price', 'quantity')->get();
        $suppliers = Supplier::where('tenant_id', $tenantId)->select('id', 'name')->get();

        return Inertia::render('Purchases/Create', [
            'products' => $products,
            'suppliers' => $suppliers,
        ]);
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(StorePurchaseRequest $request)
    {
        $this->authorizeAnyPermission(['create-purchases', 'manage-purchases']);

        return DB::transaction(function () use ($request) {
            // Find the supplier and lock for update to prevent race conditions
            $supplier = Supplier::where('id', $request->supplier_id)
                ->where('tenant_id', Auth::user()->tenant_id)
                ->lockForUpdate()
                ->first();
                
            if (!$supplier) {
                return back()->withErrors(['supplier_id' => 'Supplier not found']);
            }
            
            // Create the purchase record
            $purchase = new Purchase([
                'tenant_id' => Auth::user()->tenant_id,
                'supplier_id' => $supplier->id,
                'user_id' => Auth::id(),
                'total_amount' => 0, // Will be calculated from items
                'status' => $request->status ?? 'completed',
                'notes' => $request->notes,
            ]);
            
            $purchase->save();
            
            $totalAmount = 0;
            
            // Process each purchase item
            foreach ($request->items as $item) {
                // Find the product and lock for update
                $product = Product::where('id', $item['product_id'])
                    ->where('tenant_id', Auth::user()->tenant_id)
                    ->lockForUpdate()
                    ->first();
                    
                if (!$product) {
                    throw new \Exception("Product not found: {$item['product_id']}");
                }
                
                $quantity = $item['quantity'];
                $costPrice = $item['cost_price'];
                $subtotal = $quantity * $costPrice;
                
                // Create the purchase item
                $purchaseItem = new PurchaseItem([
                    'tenant_id' => Auth::user()->tenant_id,
                    'purchase_id' => $purchase->id,
                    'product_id' => $product->id,
                    'quantity' => $quantity,
                    'cost_price' => $costPrice,
                    'subtotal' => $subtotal,
                ]);
                
                $purchaseItem->save();
                
                // Update product quantity and purchase price
                $product->quantity += $quantity;
                
                // Update the product's purchase price with the new cost price
                $product->purchase_price = $costPrice;
                
                // Create inventory layer for FIFO tracking
                $product->addInventoryLayer(
                    $purchase->id,
                    $purchaseItem->id,
                    $quantity,
                    $costPrice
                );
                
                $product->save();
                
                $totalAmount += $subtotal;
            }
            
            // Update the purchase total
            $purchase->total_amount = $totalAmount;
            $purchase->save();
            
            // Update supplier balance (we owe the supplier more)
            $supplier->balance += $totalAmount;
            $supplier->save();
            
            return redirect()->route('purchases.index')
                ->with('success', 'Purchase created successfully.');
        });
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        $this->authorizeAnyPermission(['view-purchases', 'manage-purchases']);

        $purchase = Purchase::where('id', $id)
            ->where('tenant_id', Auth::user()->tenant_id)
            ->with(['supplier', 'items.product', 'user'])
            ->first();
            
        if (!$purchase) {
            return redirect()->route('purchases.index')
                ->with('error', 'Purchase not found');
        }
        
        // Enhance purchase items with inventory layer information
        foreach ($purchase->items as $item) {
            // Get inventory layers for this product
            $inventoryLayers = \App\Models\InventoryLayer::where('product_id', $item->product_id)
                ->where('purchase_id', $purchase->id)
                ->get();
                
            if ($inventoryLayers->isNotEmpty()) {
                $layer = $inventoryLayers->first();
                
                // Calculate percentage remaining
                $percentageRemaining = ($layer->quantity_remaining / $layer->quantity_initial) * 100;
                $item->inventory_percentage = round($percentageRemaining, 1);
                
                // Set status text based on percentage
                if ($percentageRemaining >= 90) {
                    $item->inventory_status = 'Mostly Available';
                } elseif ($percentageRemaining >= 50) {
                    $item->inventory_status = 'Partially Consumed';
                } elseif ($percentageRemaining > 0) {
                    $item->inventory_status = 'Nearly Depleted';
                } else {
                    $item->inventory_status = 'Fully Consumed';
                }
            } else {
                // No layers found (might be old data before FIFO implementation)
                $item->inventory_percentage = 0;
                $item->inventory_status = 'No Layer Data';
            }
        }
        
        return Inertia::render('Purchases/Show', [
            'purchase' => $purchase,
            'can' => [
                'manage_purchases' => auth('web')->user() ? auth('web')->user()->canAny(['manage-purchases']) : false,
                'process_purchase_returns' => auth('web')->user() ? auth('web')->user()->canAny(['process-purchase-returns']) : false,
            ]
        ]);
    }
}
