<?php

namespace App\Http\Controllers\Web;

use App\Http\Controllers\Controller;
use App\Http\Requests\StoreProductRequest;
use App\Http\Requests\UpdateProductRequest;
use App\Models\Product;
use App\Models\Category;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Gate;
use Inertia\Inertia;
use App\Traits\PermissionCheckTrait;

class ProductController extends Controller
{
    use PermissionCheckTrait;

    /**
     * Search for products by name or SKU
     * 
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function search(Request $request)
    {
        $search = $request->input('search');
        
        $products = Product::where('tenant_id', Auth::user()->tenant_id)
            ->where(function($query) use ($search) {
                $query->where('name', 'like', "%{$search}%")
                      ->orWhere('sku', 'like', "%{$search}%");
            })
            ->select('id', 'name', 'sku', 'selling_price', 'stock_quantity')
            ->limit(10)
            ->get();
            
        return response()->json(['data' => $products]);
    }
    
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $this->authorizeAnyPermission(['view-products', 'manage-products']);

        $user = Auth::user();
        $tenantId = $user->tenant_id;

        $query = Product::with('category')->where('tenant_id', $tenantId);

        // Simple search functionality
        if ($request->has('search')) {
            $searchTerm = $request->input('search');
            $query->where(function ($q) use ($searchTerm) {
                $q->where('name', 'like', "%{$searchTerm}%")
                  ->orWhere('code', 'like', "%{$searchTerm}%");
            });
        }
        
        // Category filter
        if ($request->filled('category_id')) {
            $query->where('category_id', $request->input('category_id'));
        }

        $products = $query->orderBy('name')->paginate(15)->withQueryString();

        // Determine UI capabilities using the standard Spatie 'canAny'
        $canCreate = $user->canAny(['create-products', 'manage-products']);
        $canEdit = $user->canAny(['edit-products', 'manage-products']);
        $canDelete = $user->canAny(['delete-products', 'manage-products']);

        // Get active categories for filter dropdown
        $categories = Category::where('tenant_id', $tenantId)
            ->where('active', true)
            ->orderBy('name')
            ->get();
            
        return Inertia::render('Products/Index', [
            'products' => $products,
            'categories' => $categories,
            'can' => [
                'create_product' => $canCreate,
                'edit_product' => $canEdit,
                'delete_product' => $canDelete,
            ],
            'filters' => $request->only(['search', 'category_id']),
             // Debug info can be useful
            'debug' => [
                'permissions' => $user->getAllPermissions()->pluck('name')->toArray(),
                'user_id' => $user->id,
                'user_role' => $user->role,
            ]
        ]);
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $this->authorizeAnyPermission(['create-products', 'manage-products']);
        
        // Get active categories for the dropdown
        $categories = Category::where('active', true)->get();
        
        return Inertia::render('Products/Create', [
            'categories' => $categories,
        ]);
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(StoreProductRequest $request)
    {
        $this->authorizeAnyPermission(['create-products', 'manage-products']);

        $productData = $request->validated();
        $productData['tenant_id'] = Auth::user()->tenant_id;
        
        // Assign default category if none specified
        if (empty($productData['category_id'])) {
            // Find the default 'Uncategorized' category for this tenant
            $defaultCategory = Category::where('tenant_id', Auth::user()->tenant_id)
                ->where('is_default', true)
                ->first();
                
            if ($defaultCategory) {
                $productData['category_id'] = $defaultCategory->id;
            }
        }

        Product::create($productData);

        return redirect()->route('products.index')->with('success', 'Product created successfully.');
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(string $id)
    {
        $this->authorizeAnyPermission(['edit-products', 'manage-products']);

        $product = Product::where('id', $id)
            ->where('tenant_id', Auth::user()->tenant_id)
            ->firstOrFail();
            
        // Get active categories for the dropdown
        $categories = Category::where('active', true)->get();
        
        return Inertia::render('Products/Edit', [
            'product' => $product,
            'categories' => $categories,
        ]);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(UpdateProductRequest $request, string $id)
    {
        $this->authorizeAnyPermission(['edit-products', 'manage-products']);

        $product = Product::where('id', $id)
            ->where('tenant_id', Auth::user()->tenant_id)
            ->firstOrFail();
        
        $productData = $request->validated();
        
        // If category_id is null or empty and this is removing a category
        if (array_key_exists('category_id', $productData) && empty($productData['category_id'])) {
            // Find the default 'Uncategorized' category for this tenant
            $defaultCategory = Category::where('tenant_id', Auth::user()->tenant_id)
                ->where('is_default', true)
                ->first();
                
            if ($defaultCategory) {
                $productData['category_id'] = $defaultCategory->id;
            }
        }
            
        $product->update($productData);
        
        return redirect()->route('products.index')
            ->with('message', 'Product updated successfully.');
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        $this->authorizeAnyPermission(['view-products', 'manage-products']);

        $product = Product::where('id', $id)
            ->where('tenant_id', Auth::user()->tenant_id)
            ->firstOrFail();

        // Get inventory layers for this product
        $inventoryLayers = \App\Models\InventoryLayer::where('product_id', $product->id)
            ->orderBy('created_at', 'asc') // FIFO order
            ->get();

        // Get recent transactions (purchases and sales) for this product
        $purchaseItems = \App\Models\PurchaseItem::where('product_id', $product->id)
            ->with('purchase:id,created_at')
            ->orderBy('created_at', 'desc')
            ->limit(5)
            ->get()
            ->map(function($item) {
                return [
                    'id' => $item->purchase_id,
                    'type' => 'purchase',
                    'date' => $item->purchase->created_at,
                    'quantity' => $item->quantity
                ];
            });

        $saleItems = \App\Models\SaleItem::where('product_id', $product->id)
            ->with('sale:id,created_at')
            ->orderBy('created_at', 'desc')
            ->limit(5)
            ->get()
            ->map(function($item) {
                return [
                    'id' => $item->sale_id,
                    'type' => 'sale',
                    'date' => $item->sale->created_at,
                    'quantity' => $item->quantity
                ];
            });

        // Combine and sort transactions
        $transactions = $purchaseItems->concat($saleItems)
            ->sortByDesc('date')
            ->values()
            ->take(10);

        return Inertia::render('Products/Show', [
            'product' => $product,
            'inventoryLayers' => $inventoryLayers,
            'transactions' => $transactions,
            'can' => [
                'edit_products' => auth('web')->user() ? auth('web')->user()->canAny(['edit-products', 'manage-products']) : false,
                'delete_products' => auth('web')->user() ? auth('web')->user()->canAny(['delete-products', 'manage-products']) : false,
            ]
        ]);
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        $this->authorizeAnyPermission(['delete-products', 'manage-products']);

        $product = Product::where('id', $id)
            ->where('tenant_id', Auth::user()->tenant_id)
            ->firstOrFail();
            
        $hasRelatedSaleItems = $product->saleItems()->exists();
        
        if ($hasRelatedSaleItems) {
            return back()->with('error', 'Cannot delete product as it is used in sales.');
        }
        
        $product->delete();
        
        return back()->with('message', 'Product deleted successfully.');
    }
}
