<?php

namespace App\Http\Controllers\Web;

use App\Http\Controllers\Controller;
use App\Http\Requests\StoreClientPaymentRequest;
use App\Http\Requests\StoreSupplierPaymentRequest;
use App\Models\Client;
use App\Models\Payment;
use App\Models\Sale;
use App\Models\Supplier;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Inertia\Inertia;

class PaymentController extends Controller
{
    /**
     * Store a new payment for a client.
     */
    public function storeClientPayment(Request $request, Client $client)
    {
        $validated = $request->validate([
            'amount' => 'required|numeric|min:0.01',
            'payment_date' => 'required|date',
            'method' => 'required|in:cash,bank_transfer,check',
            'bank_transfer_reference' => 'required_if:method,bank_transfer',
            'bank_transfer_receipt_path' => 'nullable|string',
            'check_number' => 'required_if:method,check',
            'check_due_date' => 'required_if:method,check|date|nullable',
            'notes' => 'nullable|string',
        ]);

        return DB::transaction(function () use ($validated, $client, $request) {
            // Lock the client for update to prevent race conditions
            $client = Client::where('id', $client->id)
                ->where('tenant_id', Auth::user()->tenant_id)
                ->lockForUpdate()
                ->firstOrFail();
            
            // Create the payment record
            $payment = new Payment([
                'tenant_id' => Auth::user()->tenant_id,
                'paymentable_type' => Client::class,
                'paymentable_id' => $client->id,
                'user_id' => Auth::id(),
                'amount' => $validated['amount'],
                'method' => $validated['method'],
                'payment_date' => $validated['payment_date'],
                'notes' => $validated['notes'] ?? null,
            ]);
            
            // Add method-specific fields
            if ($validated['method'] === 'bank_transfer') {
                $payment->bank_transfer_reference = $validated['bank_transfer_reference'];
                $payment->bank_transfer_receipt_path = $validated['bank_transfer_receipt_path'] ?? null;
            } else if ($validated['method'] === 'check') {
                $payment->check_number = $validated['check_number'];
                $payment->check_due_date = $validated['check_due_date'];
                $payment->check_status = 'pending'; // Default status for new checks
            }
            
            // Add method-specific fields
            if ($validated['method'] === 'bank_transfer') {
                $payment->bank_transfer_reference = $validated['bank_transfer_reference'];
                $payment->bank_transfer_receipt_path = $validated['bank_transfer_receipt_path'] ?? null;
            } else if ($validated['method'] === 'check') {
                $payment->check_number = $validated['check_number'];
                $payment->check_due_date = $validated['check_due_date'];
            }
            
            // Update client balance (client owes less)
            // For check payments, only reduce balance if the check should count in totals
            if ($payment->shouldCountInTotals()) {
                $client->balance -= $validated['amount'];
                $client->save();
            }
            
            // Save the payment
            $payment->save();
            
            if ($request->wantsJson()) {
                return response()->json($payment, 201);
            }
            
            return redirect()->back()->with('success', 'Payment recorded successfully.');
        });
    }
    
    /**
     * Store a new payment for a supplier.
     */
    public function storeSupplierPayment(Request $request, Supplier $supplier)
    {
        $validated = $request->validate([
            'amount' => 'required|numeric|min:0.01',
            'payment_date' => 'required|date',
            'method' => 'required|in:cash,bank_transfer,check',
            'bank_transfer_reference' => 'required_if:method,bank_transfer',
            'bank_transfer_receipt_path' => 'nullable|string',
            'check_number' => 'required_if:method,check',
            'check_due_date' => 'required_if:method,check|date|nullable',
            'notes' => 'nullable|string',
        ]);

        return DB::transaction(function () use ($validated, $supplier, $request) {
            // Lock the supplier for update to prevent race conditions
            $supplier = Supplier::where('id', $supplier->id)
                ->where('tenant_id', Auth::user()->tenant_id)
                ->lockForUpdate()
                ->firstOrFail();
            
            // Create the payment record
            $payment = new Payment([
                'tenant_id' => Auth::user()->tenant_id,
                'paymentable_type' => Supplier::class,
                'paymentable_id' => $supplier->id,
                'user_id' => Auth::id(),
                'amount' => $validated['amount'],
                'method' => $validated['method'],
                'payment_date' => $validated['payment_date'],
                'notes' => $validated['notes'] ?? null,
            ]);
            
            // Add method-specific fields
            if ($validated['method'] === 'bank_transfer') {
                $payment->bank_transfer_reference = $validated['bank_transfer_reference'];
                $payment->bank_transfer_receipt_path = $validated['bank_transfer_receipt_path'] ?? null;
            } else if ($validated['method'] === 'check') {
                $payment->check_number = $validated['check_number'];
                $payment->check_due_date = $validated['check_due_date'];
                $payment->check_status = 'pending'; // Default status for new checks
            }
            
            // Update supplier balance (supplier is owed less / we owe less)
            // For check payments, only reduce balance if the check should count in totals
            if ($payment->shouldCountInTotals()) {
                $supplier->balance -= $validated['amount'];
                $supplier->save();
            }
            
            // Save the payment
            $payment->save();
            
            if ($request->wantsJson()) {
                return response()->json($payment, 201);
            }
            
            return redirect()->back()->with('success', 'Payment recorded successfully.');
        });
    }

    /**
     * Show the form for creating a new sale payment.
     */
    public function create(Request $request)
    {
        $saleId = $request->query('sale_id');
        
        if (!$saleId) {
            return redirect()->route('sales.index')->with('error', 'Sale ID is required.');
        }

        $sale = Sale::withoutTrashed()
            ->where('id', $saleId)
            ->where('tenant_id', Auth::user()->tenant_id)
            ->with('client')
            ->firstOrFail();

        return Inertia::render('Payments/Create', [
            'sale' => $sale,
        ]);
    }

    /**
     * Store a new payment for a sale.
     */
    public function storeSalePayment(Request $request)
    {
        $validated = $request->validate([
            'sale_id' => 'required|exists:sales,id',
            'amount' => 'required|numeric|min:0.01',
            'payment_date' => 'required|date',
            'method' => 'required|in:cash,card,bank_transfer,check',
            'check_number' => 'required_if:method,check',
            'check_due_date' => 'required_if:method,check|date|nullable',
            'notes' => 'nullable|string',
        ]);

        return DB::transaction(function () use ($validated, $request) {
            // Lock the sale for update to prevent race conditions
            $sale = Sale::withoutTrashed()
                ->where('id', $validated['sale_id'])
                ->where('tenant_id', Auth::user()->tenant_id)
                ->lockForUpdate()
                ->firstOrFail();
            
            // Check if payment amount doesn't exceed remaining balance
            $remainingBalance = $sale->final_amount - $sale->paid_amount;
            if ($validated['amount'] > $remainingBalance) {
                throw new \Exception('Payment amount cannot exceed remaining balance.');
            }
            
            // Create the payment record
            $payment = new Payment([
                'tenant_id' => Auth::user()->tenant_id,
                'paymentable_type' => Sale::class,
                'paymentable_id' => $sale->id,
                'sale_id' => $sale->id,
                'user_id' => Auth::id(),
                'amount' => $validated['amount'],
                'method' => $validated['method'],
                'payment_date' => $validated['payment_date'],
                'payment_type' => 'sale_payment',
                'notes' => $validated['notes'] ?? null,
            ]);
            
            // Add method-specific fields
            if ($validated['method'] === 'check') {
                $payment->check_number = $validated['check_number'];
                $payment->check_due_date = $validated['check_due_date'];
                $payment->check_status = 'pending'; // Default status for new checks
            }
            
            // Update sale paid amount and payment status
            $sale->paid_amount += $validated['amount'];
            
            if ($sale->paid_amount >= $sale->final_amount) {
                $sale->payment_status = 'paid';
            } else if ($sale->paid_amount > 0) {
                $sale->payment_status = 'partial_paid';
            } else {
                $sale->payment_status = 'not_paid';
            }
            
            $sale->save();
            $payment->save();
            
            if ($request->wantsJson()) {
                return response()->json($payment, 201);
            }
            
            return redirect()->route('sales.index')->with('success', 'Payment recorded successfully.');
        });
    }
}
