<?php

namespace App\Http\Controllers\Web;

use App\Http\Controllers\Controller;
use App\Http\Requests\StoreExpenseRequest;
use App\Http\Requests\UpdateExpenseRequest;
use App\Models\Expense;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Inertia\Inertia;
use App\Traits\PermissionCheckTrait;

class ExpenseController extends Controller
{
    use PermissionCheckTrait;

    /**
     * Create a new controller instance.
     */
    public function __construct()
    {
        // No middleware needed here, permissions checked per method using the trait
    }

    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        // Check for any of these permissions with various formats
        $this->authorizeAnyPermission([
            'view-expenses', 
            'view-expenses',
            'manage-expenses', 
            'manage-expenses',
            'view expense', 
            'view-expense',
            'manage expense', 
            'manage-expense'
        ]);

        $user = Auth::user();
        $tenantId = $user->tenant_id;

        $expenses = Expense::where('tenant_id', $tenantId)
            ->with('user:id,name')
            ->orderBy('expense_date', 'desc')
            ->paginate(15);

        // Determine UI capabilities based on permissions using the standard Spatie 'canAny' on the user
        $canCreate = $this->userHasAnyPermission(['create-expenses', 'manage-expenses']);
        $canEdit = $this->userHasAnyPermission(['edit-expenses', 'manage-expenses']);
        $canDelete = $this->userHasAnyPermission(['delete-expenses', 'manage-expenses']);

        \Log::info('ExpenseController@index permission debug', [
            'user_id' => $user->id,
            'user_email' => $user->email,
            'canCreate' => $canCreate,
            'canEdit' => $canEdit,
            'canDelete' => $canDelete,
            'permissions' => $user->getAllPermissions()->pluck('name')->toArray(),
            'roles' => $user->getRoleNames()->toArray(),
        ]);

        return Inertia::render('Expenses/Index', [
            'expenses' => $expenses,
            'can' => [
                'create_expense' => $canCreate,
                'edit_expense' => $canEdit,
                'delete_expense' => $canDelete,
            ],
            // Debug info can be useful
            'debug' => [
                'permissions' => $user->getAllPermissions()->pluck('name')->toArray(),
                'user_id' => $user->id,
                'user_role' => $user->role, // Keep role info for context
            ]
        ]);
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        // Check for any of these permissions with various formats
        $this->authorizeAnyPermission([
            'create-expenses', 
            'create-expenses',
            'manage-expenses', 
            'manage-expenses',
            'create expense', 
            'create-expense',
            'manage expense', 
            'manage-expense'
        ]);

        return Inertia::render('Expenses/Create');
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(StoreExpenseRequest $request)
    {
        // Check for any of these permissions with various formats
        $this->authorizeAnyPermission([
            'create-expenses', 
            'create-expenses',
            'manage-expenses', 
            'manage-expenses',
            'create expense', 
            'create-expense',
            'manage expense', 
            'manage-expense'
        ]);

        $user = Auth::user();

        $expense = new Expense([
            'tenant_id' => $user->tenant_id,
            'user_id' => $user->id,
            'category' => $request->category,
            'description' => $request->description,
            'amount' => $request->amount,
            'expense_date' => $request->expense_date,
        ]);

        $expense->save();

        return redirect()->route('expenses.index')
            ->with('success', 'Expense created successfully.');
    }

    /**
     * Show the form for editing the specified resource.
     */
    /**
     * Display the specified resource.
     */
    public function show(Expense $expense)
    {
        // Check for any of these permissions with various formats
        $this->authorizeAnyPermission([
            'view-expenses', 
            'view-expenses',
            'manage-expenses', 
            'manage-expenses',
            'view expense', 
            'view-expense',
            'manage expense', 
            'manage-expense'
        ]);
        
        return Inertia::render('Expenses/Show', [
            'expense' => $expense
        ]);
    }
    
    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Expense $expense)
    {
        // Check for any of these permissions with various formats
        $this->authorizeAnyPermission([
            'edit-expenses', 
            'edit-expenses',
            'manage-expenses', 
            'manage-expenses',
            'edit expense', 
            'edit-expense',
            'manage expense', 
            'manage-expense'
        ]);

        return Inertia::render('Expenses/Edit', [
            'expense' => $expense
        ]);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(UpdateExpenseRequest $request, Expense $expense)
    {
        // Check for any of these permissions with various formats
        $this->authorizeAnyPermission([
            'edit-expenses', 
            'edit-expenses',
            'manage-expenses', 
            'manage-expenses',
            'edit expense', 
            'edit-expense',
            'manage expense', 
            'manage-expense'
        ]);

        $expense->update([
            'category' => $request->category,
            'description' => $request->description,
            'amount' => $request->amount,
            'expense_date' => $request->expense_date,
        ]);

        return redirect()->route('expenses.index')
            ->with('success', 'Expense updated successfully.');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Expense $expense)
    {
        // Check for any of these permissions with various formats
        $this->authorizeAnyPermission([
            'delete-expenses', 
            'delete-expenses',
            'manage-expenses', 
            'manage-expenses',
            'delete expense', 
            'delete-expense',
            'manage expense', 
            'manage-expense'
        ]);

        $expense->delete();

        return redirect()->route('expenses.index')
            ->with('success', 'Expense deleted successfully.');
    }
}
