<?php

namespace App\Http\Controllers\Web;

use App\Http\Controllers\Controller;
use App\Http\Requests\StoreClientRequest;
use App\Http\Requests\UpdateClientRequest;
use App\Models\Client;
use App\Models\Payment;
use App\Models\Sale;
use App\Models\SaleReturn;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Inertia\Inertia;
use App\Traits\PermissionCheckTrait;

class ClientController extends Controller
{
    use PermissionCheckTrait;

    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $this->authorizeAnyPermission(['view-clients', 'manage-clients']);

        $user = Auth::user();
        $tenantId = $user->tenant_id;

        $query = Client::where('tenant_id', $tenantId);

        if ($request->has('search')) {
            $searchTerm = $request->input('search');
            $query->where(function ($q) use ($searchTerm) {
                $q->where('name', 'like', "%{$searchTerm}%")
                  ->orWhere('email', 'like', "%{$searchTerm}%")
                  ->orWhere('phone', 'like', "%{$searchTerm}%");
            });
        }

        $clients = $query->orderBy('name')->paginate(15)->withQueryString();

        $canCreate = $user->canAny(['create-clients', 'manage-clients']);
        $canEdit = $user->canAny(['edit-clients', 'manage-clients']);
        $canDelete = $user->canAny(['delete-clients', 'manage-clients']);

        return Inertia::render('Clients/Index', [
            'clients' => $clients,
            'can' => [
                'create_client' => $canCreate,
                'edit_client' => $canEdit,
                'delete_client' => $canDelete,
            ],
            'filters' => $request->only(['search']),
            'debug' => [
                'permissions' => $user->getAllPermissions()->pluck('name')->toArray(),
                'user_id' => $user->id,
                'user_role' => $user->role,
            ]
        ]);
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $this->authorizeAnyPermission(['create-clients', 'manage-clients']);
        return Inertia::render('Clients/Create');
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $this->authorizeAnyPermission(['create-clients', 'manage-clients']);

        $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'nullable|email|max:255|unique:clients,email,NULL,id,tenant_id,' . Auth::user()->tenant_id,
            'phone' => 'nullable|string|max:20',
            'address' => 'nullable|string|max:255',
            'notes' => 'nullable|string',
            'is_default' => 'boolean',
        ]);

        if ($request->is_default) {
            Client::where('tenant_id', Auth::user()->tenant_id)
                ->where('is_default', true)
                ->update(['is_default' => false]);
        }

        $client = new Client([
            'tenant_id' => Auth::user()->tenant_id,
            'name' => $request->name,
            'email' => $request->email,
            'phone' => $request->phone,
            'address' => $request->address,
            'notes' => $request->notes,
            'balance' => 0,
            'is_default' => $request->is_default ?? false,
        ]);

        $client->save();

        return redirect()->route('clients.index')->with('success', 'Client created successfully.');
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        $this->authorizeAnyPermission(['view-clients', 'manage-clients']);

        $client = Client::where('id', $id)
            ->where('tenant_id', Auth::user()->tenant_id)
            ->firstOrFail();

        $sales = Sale::withoutTrashed()
            ->where('client_id', $client->id)
            ->with('user:id,name')
            ->orderBy('created_at', 'desc')
            ->get(['id', 'client_id', 'user_id', 'total_amount', 'discount_amount', 'final_amount', 'created_at']);
            
        $saleReturns = SaleReturn::where('client_id', $client->id)
            ->with('user:id,name')
            ->orderBy('created_at', 'desc')
            ->get();

        $payments = $client->payments()
            ->with('user:id,name')
            ->orderBy('created_at', 'desc')
            ->get();

        $transactions = collect();

        foreach ($sales as $sale) {
            // Determine the amount to use (final_amount if available, otherwise total_amount)
            $amount = isset($sale->final_amount) ? $sale->final_amount : $sale->total_amount;
            
            // Create description with discount info if applicable
            $description = 'Sale #' . $sale->id;
            if (isset($sale->discount_amount) && $sale->discount_amount > 0) {
                $description .= ' (Discount: ' . number_format($sale->discount_amount, 2) . ' MAD)';
            }
            
            $transactions->push([
                'id' => $sale->id,
                'type' => 'sale',
                'date' => $sale->created_at,
                'amount' => $amount,
                'balance_effect' => $amount, // Use final_amount for balance effect
                'description' => $description,
                'user' => $sale->user ? $sale->user->name : null,
            ]);
        }

        foreach ($saleReturns as $return) {
            $transactions->push([
                'id' => $return->id,
                'type' => 'sale_return',
                'date' => $return->created_at,
                'amount' => $return->returned_amount,
                'balance_effect' => -$return->returned_amount,
                'description' => 'Return for Sale #' . $return->sale_id,
                'user' => $return->user ? $return->user->name : null,
            ]);
        }

        foreach ($payments as $payment) {
            $description = 'Payment (' . ucfirst($payment->method) . ')';
            
            // Add status information for check payments
            if ($payment->isCheckPayment()) {
                $description .= ' - Check #' . $payment->check_number;
                if ($payment->check_status === 'unpaid') {
                    $description .= ' (Unpaid)';
                } elseif ($payment->check_status === 'pending') {
                    $description .= ' (Pending)';
                } elseif ($payment->check_status === 'paid') {
                    $description .= ' (Paid)';
                }
            }
            
            $transactions->push([
                'id' => $payment->id,
                'type' => 'payment',
                'date' => $payment->created_at,
                'amount' => $payment->amount,
                'balance_effect' => $payment->shouldCountInTotals() ? -$payment->amount : 0,
                'description' => $description,
                'user' => $payment->user ? $payment->user->name : null,
                'status' => $payment->isCheckPayment() ? $payment->check_status : 'completed',
                'is_valid' => $payment->shouldCountInTotals(),
            ]);
        }

        $transactions = $transactions->sortByDesc('date')->values();
        
        // Get the authenticated user
        $user = Auth::user();
        
        $canManageClients = $user->canAny(['manage-clients']);
        $canRecordPayments = $user->canAny(['record-payments']);

        return Inertia::render('Clients/Show', [
            'client' => $client,
            'transactions' => $transactions,
            'summary' => [
                'current_balance' => $client->balance,
                'total_sales' => $sales->sum('final_amount'),
                'total_returns' => $saleReturns->sum('returned_amount'),
                'total_payments' => $payments->filter(function ($payment) {
                    return $payment->shouldCountInTotals();
                })->sum('amount'),
            ],
            'can' => [
                'manage_clients' => $canManageClients,
                'record_payments' => $canRecordPayments,
            ],
        ]);
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(string $id)
    {
        $this->authorizeAnyPermission(['edit-clients', 'manage-clients']);

        $client = Client::where('id', $id)
            ->where('tenant_id', Auth::user()->tenant_id)
            ->firstOrFail();

        return Inertia::render('Clients/Edit', [
            'client' => $client,
        ]);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id)
    {
        $this->authorizeAnyPermission(['edit-clients', 'manage-clients']);

        $client = Client::where('id', $id)
            ->where('tenant_id', Auth::user()->tenant_id)
            ->firstOrFail();

        $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'nullable|email|max:255|unique:clients,email,' . $client->id . ',id,tenant_id,' . Auth::user()->tenant_id,
            'phone' => 'nullable|string|max:20',
            'address' => 'nullable|string|max:255',
            'notes' => 'nullable|string',
            'is_default' => 'boolean',
        ]);

        if ($request->filled('is_default') && $request->is_default && !$client->is_default) {
            Client::where('tenant_id', Auth::user()->tenant_id)
                ->where('is_default', true)
                ->update(['is_default' => false]);
        }

        $client->name = $request->name;
        $client->email = $request->email;
        $client->phone = $request->phone;
        $client->address = $request->address;
        $client->notes = $request->notes;
        $client->is_default = $request->is_default ?? false;

        $client->save();

        return redirect()->route('clients.index')->with('success', 'Client updated successfully.');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        $this->authorizeAnyPermission(['delete-clients', 'manage-clients']);

        $client = Client::where('id', $id)
            ->where('tenant_id', Auth::user()->tenant_id)
            ->firstOrFail();

        $salesCount = $client->sales()->count();
        if ($salesCount > 0) {
            return back()->with('error', 'Cannot delete client with associated sales.');
        }

        $paymentsCount = $client->payments()->count();
        if ($paymentsCount > 0) {
            return back()->with('error', 'Cannot delete client with associated payments.');
        }

        if ($client->balance > 0) {
            return back()->with('error', 'Cannot delete client with outstanding balance.');
        }

        $client->delete();

        return redirect()->route('clients.index')->with('success', 'Client deleted successfully.');
    }
}
