<?php

namespace App\Http\Controllers\Web;

use App\Http\Controllers\Controller;
use App\Models\Payment;
use Illuminate\Http\Request;
use Inertia\Inertia;

class CheckController extends Controller
{
    /**
     * Display a listing of all checks.
     */
    public function index(Request $request)
    {
        $query = Payment::where('method', 'check')
                        ->with(['paymentable', 'sale', 'user']);
        
        // Filter by client if specified
        if ($request->filled('client_id')) {
            $query->where('paymentable_type', 'App\\Models\\Client')
                  ->where('paymentable_id', $request->client_id);
        }
        
        // Filter by status if specified
        if ($request->filled('status') && $request->status !== 'all') {
            $query->where('check_status', $request->status);
        }
        
        $checks = $query->orderBy('check_due_date', 'desc')
                       ->paginate(20)
                       ->withQueryString();
        
        // Get all clients for the filter dropdown
        $clients = \App\Models\Client::select('id', 'name')
                                    ->where('tenant_id', auth()->user()->tenant_id)
                                    ->groupBy('id', 'name')
                                    ->orderBy('name')
                                    ->get();
        
        // Get statistics from all checks (not filtered)
        $allChecksQuery = Payment::where('method', 'check')
                                ->with(['paymentable', 'sale', 'user']);
        
        // Apply client filter to statistics if specified
        if ($request->filled('client_id')) {
            $allChecksQuery->where('paymentable_type', 'App\\Models\\Client')
                          ->where('paymentable_id', $request->client_id);
        }
        
        $allChecks = $allChecksQuery->get();
        
        $statistics = [
            'pending_count' => $allChecks->where('check_status', 'pending')->count(),
            'paid_count' => $allChecks->where('check_status', 'paid')->count(),
            'unpaid_count' => $allChecks->where('check_status', 'unpaid')->count(),
            'overdue_count' => $allChecks->filter(function ($check) {
                return $check->check_due_date < now()->toDateString() && $check->check_status === 'pending';
            })->count(),
            'pending_amount' => $allChecks->where('check_status', 'pending')->sum('amount'),
            'paid_amount' => $allChecks->where('check_status', 'paid')->sum('amount'),
            'unpaid_amount' => $allChecks->where('check_status', 'unpaid')->sum('amount'),
            'overdue_amount' => $allChecks->filter(function ($check) {
                return $check->check_due_date < now()->toDateString() && $check->check_status === 'pending';
            })->sum('amount'),
        ];
        
        return Inertia::render('Checks/Index', [
            'checks' => $checks->through(function ($check) {
                return [
                    'id' => $check->id,
                    'amount' => $check->amount,
                    'check_number' => $check->check_number,
                    'check_due_date' => $check->check_due_date,
                    'check_status' => $check->check_status,
                    'payment_date' => $check->payment_date,
                    'paymentable_type' => $check->paymentable_type,
                    'paymentable' => $check->paymentable ? [
                        'id' => $check->paymentable->id,
                        'name' => $check->paymentable->name,
                    ] : null,
                    'sale' => $check->sale ? [
                        'id' => $check->sale->id,
                        'final_amount' => $check->sale->final_amount,
                    ] : null,
                    'user' => [
                        'id' => $check->user->id,
                        'name' => $check->user->name,
                    ],
                    'notes' => $check->notes,
                    'is_overdue' => $check->check_due_date < now()->toDateString() && $check->check_status === 'pending',
                ];
            }),
            'clients' => $clients,
            'statistics' => $statistics,
            'filters' => [
                'client_id' => $request->client_id,
                'status' => $request->status ?? 'all',
            ],
        ]);
    }
    
    /**
     * Update check status.
     */
    public function updateStatus(Request $request, Payment $payment)
    {
        $request->validate([
            'status' => 'required|in:paid,pending,unpaid',
        ]);
        
        if (!$payment->isCheckPayment()) {
            return back()->withErrors(['error' => 'This payment is not a check payment.']);
        }
        
        $oldStatus = $payment->check_status;
        $newStatus = $request->status;
        
        // Update the payment status
        $payment->update(['check_status' => $newStatus]);
        
        // Adjust client balance based on status change
        if ($payment->paymentable_type === 'App\\Models\\Client') {
            $client = $payment->paymentable;
            
            // If changing from paid/pending to unpaid, increase client balance (they owe more)
            if (in_array($oldStatus, ['paid', 'pending']) && $newStatus === 'unpaid') {
                $client->balance += $payment->amount;
                $client->save();
            }
            // If changing from unpaid to paid/pending, decrease client balance (they owe less)
            elseif ($oldStatus === 'unpaid' && in_array($newStatus, ['paid', 'pending'])) {
                $client->balance -= $payment->amount;
                $client->save();
            }
        }
        
        // Update related sale totals if needed
        if ($payment->sale) {
            $this->updateSalePaidAmount($payment->sale);
        }
        
        return back()->with('success', 'Check status updated successfully.');
    }
    
    /**
     * Get due checks for dashboard alerts.
     */
    public function getDueChecks()
    {
        $dueChecks = Payment::getDueChecks();
        
        return response()->json([
            'due_checks' => $dueChecks->map(function ($check) {
                return [
                    'id' => $check->id,
                    'amount' => $check->amount,
                    'check_number' => $check->check_number,
                    'check_due_date' => $check->check_due_date,
                    'paymentable_name' => $check->paymentable ? $check->paymentable->name : 'Unknown',
                    'days_overdue' => now()->diffInDays($check->check_due_date, false),
                ];
            }),
        ]);
    }
    
    /**
     * Update sale paid amount based on valid payments.
     */
    private function updateSalePaidAmount($sale)
    {
        $validPayments = $sale->payments()->get()->filter(function ($payment) {
            return $payment->shouldCountInTotals();
        });
        
        $totalPaid = $validPayments->sum('amount');
        
        $sale->update(['paid_amount' => $totalPaid]);
    }
}