<?php

namespace App\Http\Controllers\Tenant;

use App\Http\Controllers\Controller;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Auth;
use Illuminate\Validation\Rule;
use Inertia\Inertia;
use Spatie\Permission\Models\Role;
use Spatie\Permission\Models\Permission;
use App\Traits\PermissionCheckTrait;

class UserController extends Controller
{
    use PermissionCheckTrait;

    /**
     * Display a listing of the users for the current tenant.
     */
    public function index(Request $request)
    {
        $this->authorizeAnyPermission(['view-users', 'manage-users']);

        $user = Auth::user();
        $tenantId = $user->tenant_id;

        $query = User::where('tenant_id', $tenantId)
                     ->where('id', '!=', $user->id); // Exclude current user

        // Search functionality
        if ($request->has('search')) {
            $searchTerm = $request->input('search');
            $query->where(function ($q) use ($searchTerm) {
                $q->where('name', 'like', "%{$searchTerm}%")
                  ->orWhere('email', 'like', "%{$searchTerm}%");
            });
        }

        $users = $query->orderBy('name')->paginate(15)->withQueryString();

        // Determine UI capabilities
        $canCreate = $user->canAny(['create-users', 'manage-users']);
        $canEdit = $user->canAny(['edit-users', 'manage-users']);
        $canDelete = $user->canAny(['delete-users', 'manage-users']);

        return Inertia::render('Tenant/Users/Index', [
            'users' => $users->through(fn ($user) => [
                'id' => $user->id,
                'name' => $user->name,
                'email' => $user->email,
                'role' => $user->roles->first()?->name, // Get the first role name
                'created_at' => $user->created_at->format('Y-m-d H:i'),
            ]),
            'can' => [
                'create_user' => $canCreate,
                'edit_user' => $canEdit,
                'delete_user' => $canDelete,
            ],
            'filters' => $request->only(['search']),
            'debug' => [
                'permissions' => $user->getAllPermissions()->pluck('name')->toArray(),
                'user_id' => $user->id,
                'user_role' => $user->roles->first()?->name,
            ]
        ]);
    }

    /**
     * Show the form for creating a new user.
     */
    public function create()
    {
        $this->authorizeAnyPermission(['create-users', 'manage-users']);

        $tenantId = auth()->user()->tenant_id;
        
        // Create both admin and cashier roles if they don't exist
        $adminRoleName = 'admin-tenant-' . $tenantId;
        $cashierRoleName = 'cashier-tenant-' . $tenantId;
        
        // Create admin role if it doesn't exist
        $adminRole = Role::firstOrCreate(
            ['name' => $adminRoleName],
            ['guard_name' => 'web']
        );
        
        // Create cashier role if it doesn't exist
        $cashierRole = Role::firstOrCreate(
            ['name' => $cashierRoleName],
            ['guard_name' => 'web']
        );
        
        // Always ensure admin role has all tenant-level permissions
        // This runs every time the create page is loaded, ensuring the role is always up to date
        $tenantPermissions = Permission::whereNotIn('name', [
            'access-superadmin-panel', 'access-superadmin-panel', 'manage-tenants'
        ])->get();
        $adminRole->syncPermissions($tenantPermissions);
        
        // Log the permissions assigned to the admin role
        \Illuminate\Support\Facades\Log::info('Admin role permissions updated', [
            'role' => $adminRole->name,
            'permissions_count' => $tenantPermissions->count(),
            'permissions' => $tenantPermissions->pluck('name')->toArray()
        ]);
        
        // Ensure cashier role has basic permissions
        if ($cashierRole->wasRecentlyCreated) {
            $cashierPermissions = [
                'view-products', 'view-clients', 'view-suppliers',
                'view-sales', 'manage-sales', 'process-sales',
                'view-dashboard', 'record-payments'
                // 'view-expenses' removed - cashiers should not see expenses by default
            ];
            $cashierRole->syncPermissions($cashierPermissions);
        }
        
        // Fetch roles available for the current tenant
        $roles = Role::where(function($query) use ($tenantId) {
            $query->where('name', 'like', '%-tenant-' . $tenantId);
        })->get();
        
        // Log the roles for debugging
        \Illuminate\Support\Facades\Log::info('Available roles for tenant', [
            'tenant_id' => $tenantId,
            'roles' => $roles->pluck('name')->toArray()
        ]);
        
        // Fetch all permissions
        $permissions = Permission::all();
        
        return Inertia::render('Tenant/Users/Create', [
            'roles' => $roles,
            'permissions' => $permissions,
        ]);
    }

    /**
     * Store a newly created user in storage.
     */
    public function store(Request $request)
    {
        $this->authorizeAnyPermission(['create-users', 'manage-users']);

        $request->validate([
            'name' => 'required|string|max:255',
            'email' => [
                'required',
                'string',
                'email',
                'max:255',
                Rule::unique('users')->where(function ($query) {
                    return $query->where('tenant_id', Auth::user()->tenant_id);
                }),
            ],
            'password' => 'required|string|min:8|confirmed',
            'permissions' => 'nullable|array',
        ]);

        $tenantId = Auth::user()->tenant_id;
        $currentUser = Auth::user();

        // Filter out any superadmin permissions if the current user is not a superadmin
        $filteredPermissions = [];
        if ($request->has('permissions') && is_array($request->permissions)) {
            $superadminPermissions = [
                'access-superadmin-panel',
                'assign-superadmin-permissions',
                'manage-tenants'
            ];
            
            foreach ($request->permissions as $permission) {
                // Skip superadmin permissions for non-superadmin users
                if (!$currentUser->is_superadmin && in_array($permission, $superadminPermissions)) {
                    continue;
                }
                $filteredPermissions[] = $permission;
            }
        }

        $user = User::create([
            'name' => $request->name,
            'email' => $request->email,
            'password' => Hash::make($request->password),
            'tenant_id' => $tenantId,
            'is_superadmin' => false, // Tenant users are never superadmins
        ]);

        // Set role based on permissions
        // If user has comprehensive permissions, they are effectively an admin
        $isAdmin = count($filteredPermissions) >= 10 && (
            in_array('manage-products', $filteredPermissions) ||
            in_array('manage-sales', $filteredPermissions) ||
            in_array('manage-users', $filteredPermissions)
        );

        // Set legacy role field (for backward compatibility)
        $user->role = $isAdmin ? 'admin' : 'cashier';
        $user->save();

        // Assign filtered permissions directly to the user
        if (!empty($filteredPermissions)) {
            $user->syncPermissions($filteredPermissions);
        }
        
        return redirect()->route('tenant.users.index')
            ->with('success', 'User created successfully.');
    }

    /**
     * Show the form for editing the specified user.
     */
    public function edit(User $user)
    {
        $this->authorizeAnyPermission(['edit-users', 'manage-users']);

        // Ensure the user belongs to the current tenant
        if ($user->tenant_id !== auth()->user()->tenant_id) {
            abort(403, 'Unauthorized action.');
        }
        
        // Get the current tenant ID
        $tenantId = auth()->user()->tenant_id;
        
        // Fetch all permissions
        $permissions = Permission::whereNotIn('name', [
            'access-superadmin-panel', 'assign-superadmin-permissions', 'manage-tenants'
        ])->get();
        
        // Get ALL permissions this user has (both direct and role-based)
        $allUserPermissions = $user->getAllPermissions()->pluck('name');
        
        return Inertia::render('Tenant/Users/Edit', [
            'user' => [
                'id' => $user->id,
                'name' => $user->name,
                'email' => $user->email,
                'role' => $user->role,
                'permissions' => $allUserPermissions,
            ],
            'permissions' => $permissions,
            'permissionGroups' => [
                'products' => 'Product Management',
                'sales' => 'Sales Management',
                'purchases' => 'Purchase Management',
                'clients' => 'Client Management',
                'suppliers' => 'Supplier Management',
                'expenses' => 'Expense Management',
                'payments' => 'Payment Management',
                'reports' => 'Reports Access',
                'users' => 'User Management',
                'dashboard' => 'Dashboard Access'
            ]
        ]);
    }

    /**
     * Update the specified user in storage.
     */
    public function update(Request $request, User $user)
    {
        $this->authorizeAnyPermission(['edit-users', 'manage-users']);

        // Ensure the user belongs to the current tenant
        if ($user->tenant_id !== auth()->user()->tenant_id) {
            abort(403, 'Unauthorized action.');
        }

        $request->validate([
            'name' => 'required|string|max:255',
            'email' => [
                'required',
                'string',
                'email',
                'max:255',
                Rule::unique('users')->where(function ($query) use ($user) {
                    return $query->where('tenant_id', Auth::user()->tenant_id);
                })->ignore($user->id),
            ],
            'password' => 'nullable|string|min:8|confirmed',
            'permissions' => 'nullable|array',
        ]);

        $tenantId = Auth::user()->tenant_id;
        $currentUser = Auth::user();

        // Filter out any superadmin permissions if the current user is not a superadmin
        $filteredPermissions = [];
        if ($request->has('permissions') && is_array($request->permissions)) {
            $superadminPermissions = [
                'access-superadmin-panel',
                'assign-superadmin-permissions',
                'manage-tenants'
            ];
            
            foreach ($request->permissions as $permission) {
                // Skip superadmin permissions for non-superadmin users
                if (!$currentUser->is_superadmin && in_array($permission, $superadminPermissions)) {
                    continue;
                }
                $filteredPermissions[] = $permission;
            }
        }

        // Update user details
        $user->name = $request->name;
        $user->email = $request->email;
        
        if ($request->filled('password')) {
            $user->password = Hash::make($request->password);
        }
        
        // Set role based on permissions
        // If user has comprehensive permissions, they are effectively an admin
        $isAdmin = count($filteredPermissions) >= 10 && (
            in_array('manage-products', $filteredPermissions) ||
            in_array('manage-sales', $filteredPermissions) ||
            in_array('manage-users', $filteredPermissions)
        );

        // Set legacy role field (for backward compatibility)
        $user->role = $isAdmin ? 'admin' : 'cashier';
        $user->save();

        // Clear all roles and permissions first
        $user->roles()->detach();
        $user->permissions()->detach();
        
        // Assign filtered permissions directly to the user
        if (!empty($filteredPermissions)) {
            $user->syncPermissions($filteredPermissions);
        }
        
        return redirect()->route('tenant.users.index')
            ->with('success', 'User updated successfully.');
    }

    /**
     * Remove the specified user from storage.
     */
    public function destroy(User $user)
    {
        $this->authorizeAnyPermission(['delete-users', 'manage-users']);

        // Ensure the user belongs to the current tenant
        if ($user->tenant_id !== auth()->user()->tenant_id) {
            abort(403, 'Unauthorized action.');
        }
        
        // Prevent self-delete
        if ($user->id === auth()->id()) {
            return redirect()->route('tenant.users.index')
                ->with('error', 'You cannot delete your own account.');
        }
        
        // Delete the user
        $user->delete();
        
        return redirect()->route('tenant.users.index')
            ->with('success', 'User deleted successfully.');
    }
}
