<?php

namespace App\Http\Controllers\Tenant;

use App\Http\Controllers\Controller;
use App\Models\StockAdjustment;
use App\Models\Product;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Inertia\Inertia;

class StockAdjustmentController extends Controller
{
    public function __construct()
    {
        $this->middleware(['auth', 'can:manage-products']);
    }
    public function index()
    {
        $adjustments = StockAdjustment::with(['product', 'user'])
            ->where('tenant_id', Auth::user()->tenant_id)
            ->orderByDesc('adjustment_date')
            ->paginate(15);
        return Inertia::render('Inventory/Adjustments/Index', [
            'adjustments' => $adjustments,
        ]);
    }

    public function create()
    {
        $products = Product::where('tenant_id', Auth::user()->tenant_id)
            ->select('id', 'name')
            ->orderBy('name')
            ->get();
        return Inertia::render('Inventory/Adjustments/Create', [
            'products' => $products,
        ]);
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'product_id' => 'required|exists:products,id',
            'adjustment_type' => 'required|in:addition,subtraction',
            'quantity' => 'required|integer|min:1',
            'reason' => 'nullable|string|max:255',
        ]);
        DB::transaction(function () use ($validated) {
            $product = Product::where('id', $validated['product_id'])
                ->where('tenant_id', Auth::user()->tenant_id)
                ->lockForUpdate()
                ->firstOrFail();
            if ($validated['adjustment_type'] === 'subtraction' && $product->quantity < $validated['quantity']) {
                abort(400, 'Not enough stock for subtraction.');
            }
            $adjustment = StockAdjustment::create([
                'tenant_id' => Auth::user()->tenant_id,
                'user_id' => Auth::id(),
                'product_id' => $product->id,
                'adjustment_type' => $validated['adjustment_type'],
                'quantity' => $validated['quantity'],
                'reason' => $validated['reason'],
                'adjustment_date' => now(),
            ]);
            if ($validated['adjustment_type'] === 'addition') {
                $product->increment('quantity', $validated['quantity']);
            } else {
                $product->decrement('quantity', $validated['quantity']);
            }
        });
        return redirect()->route('stock-adjustments.index')->with('success', 'Stock adjustment recorded successfully.');
    }
}
