<?php

namespace App\Http\Controllers\Tenant;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Inertia\Inertia;
use App\Models\Sale;
use App\Models\SaleItem;
use App\Models\Product;
use App\Models\Category;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class ProfitReportController extends Controller
{
    /**
     * Display profit report by sales.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Inertia\Response
     */
    public function salesProfit(Request $request)
    {
        // Validate request parameters
        $request->validate([
            'start_date' => 'nullable|date',
            'end_date' => 'nullable|date|after_or_equal:start_date',
        ]);

        // Get date range from request or default to last 30 days
        $startDate = $request->input('start_date', now()->subDays(30)->startOfDay());
        $endDate = $request->input('end_date', now()->endOfDay());

        // Query sales within date range for current tenant (exclude soft deleted)
        $sales = Sale::withoutTrashed()
            ->where('tenant_id', Auth::user()->tenant_id)
            ->whereBetween('created_at', [$startDate, $endDate])
            ->with(['client', 'saleItems'])
            ->get();

        // Calculate profit metrics for each sale
        $salesProfitData = $sales->map(function ($sale) {
            // Calculate total COGS (Cost of Goods Sold)
            $totalCogs = $sale->saleItems->sum(function ($item) {
                return $item->quantity * ($item->cost_price_at_sale ?? 0);
            });

            // Get total revenue from final_amount (after any discounts)
            $totalRevenue = $sale->final_amount;
            
            // Calculate gross profit
            $grossProfit = $totalRevenue - $totalCogs;
            
            // Calculate profit margin percentage (if revenue > 0)
            $profitMarginPercentage = $totalRevenue > 0 
                ? round(($grossProfit / $totalRevenue) * 100, 2) 
                : 0;

            return [
                'id' => $sale->id,
                'date' => $sale->created_at->format('Y-m-d H:i'),
                'client_name' => $sale->client->name,
                'total_revenue' => $totalRevenue,
                'total_cogs' => $totalCogs,
                'gross_profit' => $grossProfit,
                'profit_margin_percentage' => $profitMarginPercentage,
            ];
        });

        // Calculate totals
        $totals = [
            'total_revenue' => $salesProfitData->sum('total_revenue'),
            'total_cogs' => $salesProfitData->sum('total_cogs'),
            'total_gross_profit' => $salesProfitData->sum('gross_profit'),
            'avg_profit_margin' => $salesProfitData->avg('profit_margin_percentage'),
        ];

        // Return Inertia view with data
        return Inertia::render('Reports/Profit/SalesProfit', [
            'salesProfitData' => $salesProfitData,
            'totals' => $totals,
            'filters' => [
                'start_date' => $startDate,
                'end_date' => $endDate,
            ],
        ]);
    }

    /**
     * Display profit report by products.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Inertia\Response
     */
    public function productsProfit(Request $request)
    {
        // Validate request parameters
        $request->validate([
            'start_date' => 'nullable|date',
            'end_date' => 'nullable|date|after_or_equal:start_date',
            'product_id' => 'nullable|exists:products,id',
            'category_id' => 'nullable|exists:categories,id',
        ]);

        // Get date range from request or default to last 30 days
        $startDate = $request->input('start_date', now()->subDays(30)->startOfDay());
        $endDate = $request->input('end_date', now()->endOfDay());

        // Start building the query for sale items
        $query = SaleItem::whereHas('sale', function ($query) use ($startDate, $endDate) {
            $query->where('tenant_id', Auth::user()->tenant_id)
                ->whereBetween('created_at', [$startDate, $endDate]);
        })->with('product');

        // Apply product filter if provided
        if ($request->filled('product_id')) {
            $query->where('product_id', $request->input('product_id'));
        }

        // Apply category filter if provided
        if ($request->filled('category_id')) {
            $query->whereHas('product', function ($query) use ($request) {
                $query->where('category_id', $request->input('category_id'));
            });
        }

        // Execute query and group by product
        $saleItems = $query->get();
        $productGroups = $saleItems->groupBy('product_id');

        // Calculate profit metrics for each product
        $productsProfitData = $productGroups->map(function ($items, $productId) {
            $product = $items->first()->product;
            
            // Calculate total quantity sold
            $totalQuantity = $items->sum('quantity');
            
            // Calculate total revenue (price_at_sale * quantity for each item)
            $totalRevenue = $items->sum(function ($item) {
                return $item->price_at_sale * $item->quantity;
            });
            
            // Calculate total COGS
            $totalCogs = $items->sum(function ($item) {
                return ($item->cost_price_at_sale ?? 0) * $item->quantity;
            });
            
            // Calculate gross profit
            $grossProfit = $totalRevenue - $totalCogs;
            
            // Calculate profit margin percentage
            $profitMarginPercentage = $totalRevenue > 0 
                ? round(($grossProfit / $totalRevenue) * 100, 2) 
                : 0;

            return [
                'product_id' => $productId,
                'product_name' => $product->name,
                'product_sku' => $product->sku,
                'category' => $product->category ? $product->category->name : 'No Category',
                'total_quantity' => $totalQuantity,
                'total_revenue' => $totalRevenue,
                'total_cogs' => $totalCogs,
                'gross_profit' => $grossProfit,
                'profit_margin_percentage' => $profitMarginPercentage,
            ];
        })->values();

        // Calculate totals
        $totals = [
            'total_quantity' => $productsProfitData->sum('total_quantity'),
            'total_revenue' => $productsProfitData->sum('total_revenue'),
            'total_cogs' => $productsProfitData->sum('total_cogs'),
            'total_gross_profit' => $productsProfitData->sum('gross_profit'),
            'avg_profit_margin' => $productsProfitData->avg('profit_margin_percentage'),
        ];

        // Get categories for filter dropdown
        $categories = Category::where('active', true)->get();
        
        // Get products for filter dropdown
        $products = Product::where('tenant_id', Auth::user()->tenant_id)
            ->when($request->filled('category_id'), function ($query) use ($request) {
                $query->where('category_id', $request->input('category_id'));
            })
            ->get();

        // Return Inertia view with data
        return Inertia::render('Reports/Profit/ProductsProfit', [
            'productsProfitData' => $productsProfitData,
            'totals' => $totals,
            'filters' => [
                'start_date' => $startDate,
                'end_date' => $endDate,
                'product_id' => $request->input('product_id'),
                'category_id' => $request->input('category_id'),
            ],
            'categories' => $categories,
            'products' => $products,
        ]);
    }
}
