<?php

namespace App\Http\Controllers\Tenant;

use App\Http\Controllers\Controller;
use App\Http\Requests\FinalizePOSSaleRequest;
use App\Models\Category;
use App\Models\Client;
use App\Models\Payment;
use App\Models\Product;
use App\Models\Sale;
use App\Models\SaleItem;
use App\Models\Tenant;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Inertia\Inertia;

class POSController extends Controller
{
    /**
     * Display the POS interface.
     *
     * @return \Inertia\Response
     */
    public function index()
    {
        // Ensure user has permission to process sales
        if (!Auth::user()->can('process-sales')) {
            abort(403, 'Unauthorized action.');
        }

        // Get tenant information
        $tenant = Tenant::find(Auth::user()->tenant_id);
        
        // Get client list
        $clients = Client::where('tenant_id', Auth::user()->tenant_id)
            ->select('id', 'name', 'email', 'phone', 'balance')
            ->orderBy('name')
            ->get();
        
        // Get a default walk-in client if it exists, or create default client data
        $walkInClient = $clients->where('name', 'Walk-in Customer')->first();
        if (!$walkInClient) {
            $walkInClient = [
                'id' => null,
                'name' => 'Walk-in Customer',
                'balance' => 0
            ];
        }
        
        // Get categories for filtering
        $categories = Category::where('tenant_id', Auth::user()->tenant_id)
            ->where('active', true)
            ->select('id', 'name', 'slug')
            ->orderBy('name')
            ->get();
        
        // Initially we'll just fetch a small set of popular products
        // For large catalogs, this should be replaced with an API call
        $popularProducts = Product::where('tenant_id', Auth::user()->tenant_id)
            ->with('category:id,name')
            ->orderBy('quantity', 'desc') // Assuming higher inventory items are popular
            ->select('id', 'name', 'sku', 'barcode', 'price', 'purchase_price', 'quantity', 'category_id')
            ->limit(20)
            ->get();
        
        // Get tenant settings that might be relevant
        $settings = [
            'tenant_name' => $tenant->name,
            'currency' => $tenant->settings['currency'] ?? 'MAD',
            'logo' => $tenant->settings['logo'] ?? null,
            'allow_negative_inventory' => $tenant->settings['allow_negative_inventory'] ?? false,
        ];
        
        return Inertia::render('POS/Index', [
            'clients' => $clients,
            'walkInClient' => $walkInClient,
            'popularProducts' => $popularProducts,
            'categories' => $categories,
            'settings' => $settings,
            'user' => [
                'id' => Auth::id(),
                'name' => Auth::user()->name,
            ],
        ]);
    }
    
    /**
     * Search for products.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function searchProducts(Request $request)
    {
        $query = $request->input('query');
        $categoryId = $request->input('category_id');
        $limit = $request->input('limit', 15);
        
        $products = Product::where('tenant_id', Auth::user()->tenant_id)
            ->with('category:id,name')
            ->where(function($q) use ($query) {
                if ($query) {
                    $q->where('name', 'like', "%{$query}%")
                      ->orWhere('sku', 'like', "%{$query}%")
                      ->orWhere('barcode', 'like', "%{$query}%");
                }
            })
            ->when($categoryId, function($q) use ($categoryId) {
                $q->where('category_id', $categoryId);
            })
            ->where('quantity', '>', 0) // Only show in-stock items by default
            ->select('id', 'name', 'sku', 'barcode', 'price', 'purchase_price', 'quantity', 'category_id')
            ->limit($limit)
            ->get();
            
        return response()->json($products);
    }
    
    /**
     * Finalize a sale from the POS screen.
     *
     * @param  \App\Http\Requests\FinalizePOSSaleRequest  $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function finalizeSale(FinalizePOSSaleRequest $request)
    {
        try {
            // Use a database transaction to ensure all operations succeed or fail together
            return DB::transaction(function () use ($request) {
                // Get the authenticated user
                $user = Auth::user();
                $tenant = Tenant::find($user->tenant_id);
            
                // No TVA calculations for now per client requirements
                // We'll add this in a future update if needed
                
                // Step 1: Fetch the client and lock the row for update to prevent race conditions
                $client = Client::where('id', $request->client_id)
                    ->where('tenant_id', $user->tenant_id)
                    ->lockForUpdate()
                    ->firstOrFail();
                    
                // Step 2: Calculate sale totals
                $saleSubtotal = 0;
                
                // Prepare items for storage
                $saleItems = [];
            
                foreach ($request->items as $itemData) {
                    // Fetch the product
                    $product = Product::where('id', $itemData['product_id'])
                        ->where('tenant_id', $user->tenant_id)
                        ->firstOrFail();
                    
                    // Use provided price or fallback to product price
                    $priceAtSaleHt = $itemData['selling_price'] ?? $product->price;
                    
                    // Calculate line subtotal
                    $lineSubtotal = $priceAtSaleHt * $itemData['quantity'];
                    
                    // Calculate item discount if provided
                    $itemDiscountAmount = 0;
                    if (!empty($itemData['item_discount_type']) && !empty($itemData['item_discount_value'])) {
                        if ($itemData['item_discount_type'] === 'percentage') {
                            $itemDiscountAmount = ($lineSubtotal * $itemData['item_discount_value']) / 100;
                        } else { // fixed
                            $itemDiscountAmount = min($itemData['item_discount_value'], $lineSubtotal); // Can't discount more than the line total
                        }
                    }
                    
                    // Calculate line total after item discount
                    $lineTotal = $lineSubtotal - $itemDiscountAmount;
                    
                    // Update running totals
                    $saleSubtotal += $lineTotal;
                    
                    // Calculate cost and profit data
                    $costAtSale = $product->purchase_price ?? 0;
                    $unitProfit = $priceAtSaleHt - $costAtSale;
                    $totalCost = $costAtSale * $itemData['quantity'];
                    $totalProfit = $lineTotal - $totalCost;
                    
                    // Prepare sale item data
                    $saleItems[] = [
                        'product_id' => $product->id,
                        'quantity' => $itemData['quantity'],
                        'price_at_sale' => $priceAtSaleHt,
                        'cost_price_at_sale' => $costAtSale, // Use the correct column name for cost price
                        // These fields will be used for calculations but not stored directly
                        // They are calculated here for use in the transaction
                        // 'unit_profit' => $unitProfit,
                        // 'total_cost' => $totalCost,
                        // 'total_profit' => $totalProfit,
                    ];
                    
                    // Update product inventory
                    if (!$tenant->settings['allow_negative_inventory'] && $product->quantity < $itemData['quantity']) {
                        throw new \Exception("Insufficient stock for product: {$product->name}");
                    }
                    
                    $product->decrement('quantity', $itemData['quantity']);
                }
                
                // Step 3: Apply overall sale discount
                $totalDiscountAmount = $request->discount_amount ?? 0;
            
                // Ensure discount doesn't exceed subtotal
                $totalDiscountAmount = min($totalDiscountAmount, $saleSubtotal);
                
                // Calculate final amount (after discount, no TVA)
                $saleFinalAmount = $saleSubtotal - $totalDiscountAmount;
                
                // Step 4: Create the sale record
                $sale = Sale::create([
                    'tenant_id' => $user->tenant_id,
                    'user_id' => $user->id,
                    'client_id' => $client->id,
                    'total_amount' => $saleSubtotal, // This is the amount before discount
                    'discount_amount' => $totalDiscountAmount,
                    'final_amount' => $saleFinalAmount, // This is the final amount after discount
                    'status' => 'completed',
                ]);
                
                // Step 5: Create the sale items
                foreach ($saleItems as $item) {
                    $sale->saleItems()->create(array_merge($item, ['tenant_id' => $user->tenant_id]));
                }
                
                // Step 6: Update client balance with the final amount
                $client->balance += $saleFinalAmount;
                $client->save();
                
                // Step 7: Handle Walk-in Cash Payment if applicable
                if ($request->is_walk_in_cash_payment && ($client->name === 'Walk-in Customer' || $client->name === 'Cash Client')) {
                    // Create a cash payment record
                    Payment::create([
                        'tenant_id' => $user->tenant_id,
                        'paymentable_id' => $client->id,
                        'paymentable_type' => Client::class,
                        'user_id' => $user->id,
                        'amount' => $saleFinalAmount,
                        'method' => 'cash',
                        'payment_date' => now(),
                        'notes' => 'POS immediate cash payment',
                    ]);
                    
                    // Reset the walk-in client's balance since they paid immediately
                    $client->balance -= $saleFinalAmount;
                    $client->save();
                }
                
                // Return the created sale for receipt printing and confirmation
                return response()->json([
                    'success' => true,
                    'sale_id' => $sale->id,
                    'message' => 'Sale completed successfully.',
                    'amount' => $saleFinalAmount,
                    'receipt_url' => route('sales.receipt', $sale->id),
                ]);
            });
        } catch (\Exception $e) {
            // Log the error for debugging
            \Log::error('POS Sale Error: ' . $e->getMessage());
            
            // Return a user-friendly error message
            return response()->json([
                'success' => false,
                'message' => 'An error occurred while processing the sale: ' . $e->getMessage(),
            ], 500);
        }
    }
    
    /**
     * Generate and display a receipt for a completed sale.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function receipt($id)
    {
        $user = Auth::user();
        
        // Find the sale with related data
        $sale = Sale::with(['saleItems.product', 'client', 'user'])
            ->where('tenant_id', $user->tenant_id)
            ->where('id', $id)
            ->firstOrFail();
        
        // Get tenant settings for receipt customization
        $tenant = Tenant::find($user->tenant_id);
        $settings = $tenant->settings;
        
        // Calculate totals
        $subtotal = $sale->total_amount; // This is stored as HT (pre-discount)
        $discount = $sale->discount_amount;
        $finalAmount = $sale->final_amount; // This is stored as TTC (after discount and TVA)
        
        // Default TVA rate from tenant settings
        $tvaRate = $settings['default_tva_rate'] ?? 0;
        $tvaAmount = $finalAmount - $subtotal + $discount;
        
        return Inertia::render('Sales/Receipt', [
            'sale' => $sale,
            'subtotal' => $subtotal,
            'discount' => $discount,
            'tvaRate' => $tvaRate,
            'tvaAmount' => $tvaAmount,
            'finalAmount' => $finalAmount,
            'settings' => $settings,
            'printMode' => true,
        ]);
    }
}
