<?php

namespace App\Http\Controllers\Tenant;

use App\Http\Controllers\Controller;
use App\Models\Sale;
use Barryvdh\DomPDF\Facade\Pdf;
use Illuminate\Http\Request;
use Illuminate\Http\Response;
use Illuminate\Support\Facades\Auth;

class InvoiceController extends Controller
{
    /**
     * Generate and download a PDF invoice for a sale
     *
     * @param Sale $sale
     * @return \Illuminate\Http\Response
     */
    public function download(Sale $sale)
    {
        // Authorization check
        if ($sale->tenant_id !== Auth::user()->tenant_id) {
            abort(403, 'You are not authorized to access this sale.');
        }

        // Eager load necessary relationships
        $sale->load(['tenant', 'user', 'client', 'saleItems.product']);
        
        // Get the tenant
        $tenant = $sale->tenant;
        
        // Calculate subtotal (sum of line totals)
        $subtotal = $sale->saleItems->sum('line_total');
        
        // Generate the PDF
        $pdf = PDF::loadView('invoices.sales_default', compact('sale', 'tenant', 'subtotal'));
        
        // Set paper size and orientation
        $pdf->setPaper('a4', 'portrait');
        
        // Try a direct inline approach instead of attachment
        // This will display the PDF in the browser rather than downloading it
        return $pdf->stream("invoice-{$sale->id}.pdf", ['Attachment' => false]);
    }
}
