<?php

namespace App\Http\Controllers\Tenant;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Category;
use App\Models\Product;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Str;
use Inertia\Inertia;
use App\Traits\PermissionCheckTrait;

class CategoryController extends Controller
{
    use PermissionCheckTrait;
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $this->authorizeAnyPermission(['manage-products']);
        
        $user = Auth::user();
        $tenantId = $user->tenant_id;
        
        $query = Category::where('tenant_id', $tenantId);
        
        // Apply search filter if provided
        if ($request->filled('search')) {
            $search = $request->input('search');
            $query->where('name', 'like', "%{$search}%");
        }
        
        // Sort by default categories first, then by name
        $query->orderBy('is_default', 'desc')->orderBy('name', 'asc');
        
        $categories = $query->paginate(10)->withQueryString();
        
        return Inertia::render('Products/Categories/Index', [
            'categories' => $categories,
            'filters' => $request->only(['search']),
        ]);
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $this->authorizeAnyPermission(['manage-products']);
        return Inertia::render('Products/Categories/Create');
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $this->authorizeAnyPermission(['manage-products']);
        
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'description' => 'nullable|string',
            'active' => 'boolean',
        ]);
        
        $validated['tenant_id'] = Auth::user()->tenant_id;
        $validated['slug'] = Str::slug($validated['name']);
        $validated['is_default'] = false; // New categories can't be default
        
        Category::create($validated);
        
        return redirect()->route('categories.index')->with('success', 'Category created successfully.');
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        $this->authorizeAnyPermission(['manage-products']);
        
        $category = Category::where('id', $id)
            ->where('tenant_id', Auth::user()->tenant_id)
            ->firstOrFail();
            
        return Inertia::render('Products/Categories/Show', [
            'category' => $category,
        ]);
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(string $id)
    {
        $this->authorizeAnyPermission(['manage-products']);
        
        $category = Category::where('id', $id)
            ->where('tenant_id', Auth::user()->tenant_id)
            ->firstOrFail();
            
        return Inertia::render('Products/Categories/Edit', [
            'category' => $category,
        ]);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id)
    {
        $this->authorizeAnyPermission(['manage-products']);
        
        $category = Category::where('id', $id)
            ->where('tenant_id', Auth::user()->tenant_id)
            ->firstOrFail();
            
        // Prevent updating the default status of a default category
        if ($category->is_default) {
            $request->merge(['is_default' => true]);
        }
        
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'description' => 'nullable|string',
            'active' => 'boolean',
        ]);
        
        $validated['slug'] = Str::slug($validated['name']);
        
        $category->update($validated);
        
        return redirect()->route('categories.index')
            ->with('success', 'Category updated successfully.');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        $this->authorizeAnyPermission(['manage-products']);
        
        $category = Category::where('id', $id)
            ->where('tenant_id', Auth::user()->tenant_id)
            ->firstOrFail();
            
        // Prevent deletion of the default category
        if ($category->is_default) {
            return redirect()->route('categories.index')
                ->with('error', 'Cannot delete the default Uncategorized category.');
        }
        
        // Get the default category to reassign products
        $defaultCategory = Category::where('tenant_id', Auth::user()->tenant_id)
            ->where('is_default', true)
            ->first();
            
        if (!$defaultCategory) {
            return redirect()->route('categories.index')
                ->with('error', 'Default category not found. Cannot reassign products.');
        }
        
        // Reassign products to the default category
        Product::where('category_id', $category->id)
            ->update(['category_id' => $defaultCategory->id]);
            
        $category->delete();
        
        return redirect()->route('categories.index')
            ->with('success', 'Category deleted and products reassigned to Uncategorized category.');
    }
}
